(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 3.0,
MathReader 3.0, or any compatible application. The data for the notebook 
starts with the line of stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     92208,       2954]*)
(*NotebookOutlinePosition[     93347,       2991]*)
(*  CellTagsIndexPosition[     93303,       2987]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["A Simulator for Feynman's Quantum Computer", "Title",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["Colin P. Williams", "Subsubtitle",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["Off[General::spell1]", "Input",
  Editable->False,
  InitializationCell->True,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["Copyright Notice", "Section",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
Copyright Colin P. Williams (1997).

This Notebook is intended to be used in conjunction with \"Explorations in \
Quantum Computing\" by Colin P. Williams and Scott H. Clearwater, TELOS, \
Springer-Verlag (1997), ISBN:0-387-94768-X. Permission is hereby granted to \
copy and distribute this Notebook freely for any non-commercial activity \
provided you include this copyright notice at the beginning of all such \
copies. Please send suggestions and bug reports to Colin P. Williams at 
        colin@solstice.jpl.nasa.gov      (818) 306 6512 or 
        cpw@cs.stanford.edu               (415) 728 2118
For information on \"Explorations in Quantum Computing\" check out the TELOS \
web site:  http://www.telospub.com/catalog/PHYSICS/Explorations.html. To \
order call 1-800-777-4643.

All other rights reserved.\
\>", "Text",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Overview", "Section",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
This Notebook contains code for simulating Feynman's quantum \
computer (see \"Explorations in Quantum Computing\", Chapter 4).  It contains \
tools for building mathematical representations of quantum gates, tools for \
embedding quantum gates in quantum circuits and a simulator for the Feynman \
quantum computer that implements a given quantum circuit.   Be aware that \
classical simulations of quantum computers are very costly, computationally, \
so only simple circuits can be simulated within a reasonable time.  \
Nevertheless, the simulator is sufficient to illustrate several important \
features of quantum computation.\
\>", "Text",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
You can build any classical circuit out of the following three \
types of reversible logic gates:\
\>", "Text",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
NOTGate
CNGate
CCNGate\
\>", "Input",
  Editable->False,
  AspectRatioFixed->True],

Cell["\<\
These are the NOT gate, the CONTROLLED-NOT gate and the \
CONTROLLED-CONTROLLED-NOT gate respectively.  A particularly \"quantum\" gate \
is the one input/one output square root of NOT gate:\
\>", "Text",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
SqrtNOTGate
\
\>", "Input",
  Editable->False,
  AspectRatioFixed->True],

Cell["\<\
The SqrtNOTGate can be used to devise a circuit for computing the \
NOT function that is inherently \"quantum\" in the sense that it relies upon \
the Superposition Principle of quantum mechanics.  

Once you have designed your circuit,  you can simulate a Feynman quantum \
computer that implements this circuit using either of the commands:\
\>", 
  "Text",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
SchrodingerEvolution
EvolveQC\
\>", "Input",
  Editable->False,
  AspectRatioFixed->True],

Cell["\<\
SchrodingerEvolution evolves the quantum computer for a fixed \
period of time, EvolveQC evolves the quantum computer until the computation \
is completed.  In either case, you can visualize the resulting evolution \
using:\
\>", "Text",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["PlotEvolution", "Input",
  Editable->False,
  AspectRatioFixed->True],

Cell["\<\
See the Notebook ErrorsInQCs.ma to investigate the effects of \
errors in the operation of a quantum computer.  See the Notebook \
ErrorCorrection.ma to investigate techniques for quantum error \
correction.\
\>", "Text",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["What Computation are we going to Simulate?", "Section",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
We are going to describe a quantum circuit that computes the NOT \
function via the application of two \"square root of NOT\" gates connected \
back to back.  There is no classical gate that can achieve the SqrtNOT \
operation i.e. there is no classical gate such that two consecutive \
applications of this (hypothetical, classical) gate yield the NOT \
operation.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["Gates & their Truth Tables", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
NOT = {{0, 1},
       {1, 0}};\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell["\<\
CN = {{1,0,0,0},
      {0,1,0,0},
      {0,0,0,1},
      {0,0,1,0}};\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell["\<\
CCN = {{1,0,0,0,0,0,0,0},
       {0,1,0,0,0,0,0,0},
       {0,0,1,0,0,0,0,0},
       {0,0,0,1,0,0,0,0},
       {0,0,0,0,1,0,0,0},
       {0,0,0,0,0,1,0,0},
       {0,0,0,0,0,0,0,1},
       {0,0,0,0,0,0,1,0}};\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Creation and Annihilation Operators", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
aOP = {{0,1},   (* annihilation operator on a single bit *)
       {0,0}};
       
annihilationOP[i_, m_]:=
\tApply[Direct,
\t      ReplacePart[Table[IdentityMatrix[2], {m}], aOP, i]
\t     ]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell["\<\
cOP = {{0,0},     (* creation operator that acts on a single bit *)
       {1,0}};

creationOP[i_, m_]:=
\tApply[Direct,
\t      ReplacePart[Table[IdentityMatrix[2], {m}], cOP, i]
\t     ]
\t\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Rewriting Gates using Creation & Annihilation Operators", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
We can express the NOT, CONTROLLED-NOT and \
CONTROLLED-CONTROLLED-NOT gates as sums and products of creation and \
annihilation operators. Moreover, we can embed gates in a circuit that \
contains multiple input and output lines.  For example, NOTGate[i,m] is a NOT \
gate acting on the i-th of m inputs.  CNGate[i,j,m] is a CONTROLLED-NOT gate \
acting on the i-th and j-th of m inputs etc.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
NOTGate[i_, m_]:=
\tcreationOP[i,m] + annihilationOP[i,m]

CNGate[i_, j_, m_]:=
\t(creationOP[i,m] .
\t annihilationOP[i,m] . (annihilationOP[j,m] + creationOP[j,m]) +
\t annihilationOP[i,m] . creationOP[i,m]
\t)
\t
\t
CCNGate[i_, j_, k_, m_]:=
\t(IdentityMatrix[2^m] + 
\t  creationOP[i,m] . 
\t  annihilationOP[i,m] . 
\t  creationOP[j,m] . 
\t  annihilationOP[j,m] . 
\t   (annihilationOP[k,m] + creationOP[k,m] - IdentityMatrix[2^m])
\t)
\t\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Square Root of NOT Gate (SqrtNOT)", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["A SqrtNOT gate is a 1 input / 1 output quantum gate.", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .33333 
MathPictureStart
%% Graphics
/Courier-Bold findfont 9  scalefont  setfont
% Scaling calculations
0.34127 0.31746 0.00793651 0.31746 [
[ 0 0 0 0 ]
[ 1 .33333 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
[ ] 0 setdash
0 g
p
P
0 0 m
1 0 L
1 .33333 L
0 .33333 L
closepath
clip
newpath
p
1 Mabswid
.02381 .16667 m
.34127 .16667 L
.34127 .00794 L
.65873 .16667 L
.34127 .3254 L
.34127 .00794 L
s
.65873 .16667 m
.97619 .16667 L
s
[(SqrtNOT)] .5 .16667 0 0 Mshowa
P
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True,
  ImageSize->{248, 82},
  ImageMargins->{{34, Inherited}, {Inherited, Inherited}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40003h0000DQ000`40O003h00On7oo003hOol005Ao
o`800:9oo`00E7oo00@007oo00000:1oo`00E7oo00@007ooOomoo`8009ioo`00E7oo00<007ooOol0
0goo0P00W7oo001DOol00`00Oomoo`05Ool2002JOol005Aoo`03001oogoo00Moo`8009Qoo`00E7oo
00<007ooOol02Goo0P00UWoo001DOol00`00Oomoo`0;Ool2002DOol005Aoo`03001oogoo00eoo`80
099oo`00E7oo00<007ooOol03goo0P00T7oo001DOol00`00Oomoo`0AOol2002>Ool005Aoo`03001o
ogoo01=oo`8008aoo`00E7oo00<007ooOol05Goo0P00RWoo001DOol00`00Oomoo`0GOol20028Ool0
05Aoo`03001oogoo01Uoo`8008Ioo`00E7oo00<007ooOol06goo0P00Q7oo001DOol00`00Oomoo`0M
Ool20022Ool005Aoo`03001oogoo01moo`80081oo`00E7oo00<007ooOol08Goo0P00OWoo001DOol0
0`00Oomoo`0SOol2001lOol005Aoo`03001oogoo02Eoo`8007Yoo`00E7oo00<007ooOol09goo0P00
N7oo001DOol00`00Oomoo`0YOol2001fOol005Aoo`03001oogoo02]oo`8007Aoo`00E7oo00<007oo
Ool0;Goo0P00LWoo001DOol00`00Oomoo`0_Ool2001`Ool005Aoo`03001oogoo035oo`8006ioo`00
E7oo00<007ooOol0<goo0P00K7oo001DOol00`00Oomoo`0eOol2001ZOol005Aoo`03001oogoo03Mo
o`8006Qoo`00E7oo00<007ooOol0>Goo0P00IWoo001DOol00`00Oomoo`0kOol2001TOol005Aoo`03
001oogoo03eoo`80069oo`00E7oo00<007ooOol0?goo0P00H7oo001DOol00`00Oomoo`11Ool2001N
Ool005Aoo`03001oogoo01Qoo`<002Qoo`8005aoo`00E7oo00<007ooOol06Goo00<007ooOol0:Goo
0P00FWoo001DOol00`00Oomoo`0BOol30002Ool30002Ool40002Ool400000goo00000002Ool00`00
Ool000020002Ool3000BOol2001HOol005Aoo`03001oogoo01Eoo`80009oo`05001oogooOol00004
Ool00`00Oomoo`03Ool00`00Ool000020003Ool01000Oomoo`005Goo0P00EWoo0007Oom>000FOol3
0002Ool01@00Oomoogoo000017oo00<007ooOol00goo00<007oo00000P000goo00@007ooOol001Io
odl000Qoo`00E7oo00<007ooOol04Woo0P000Woo00@007ooOol000=oo`03001oogoo009oo`03001o
ogoo00=oo`800003Ool0000000=oo`04001oogoo000DOol2001GOol005Aoo`03001oogoo019oo`03
001oogoo009oo`<0009oo`T000=oo`800003Ool0000000=oo`800004Ool007oo000@Ool2001IOol0
05Aoo`03001oogoo01=oo`<000aoo`03001oogoo009oo`80009oo`D00003Ool0000000<000ioo`80
05]oo`00E7oo00<007ooOol08Woo00<007ooOol07Goo0P00GGoo001DOol00`00Oomoo`10Ool2001O
Ool005Aoo`03001oogoo03ioo`80065oo`00E7oo00<007ooOol0?7oo0P00Hgoo001DOol00`00Oomo
o`0jOol2001UOol005Aoo`03001oogoo03Qoo`8006Moo`00E7oo00<007ooOol0=Woo0P00JGoo001D
Ool00`00Oomoo`0dOol2001[Ool005Aoo`03001oogoo039oo`8006eoo`00E7oo00<007ooOol0<7oo
0P00Kgoo001DOol00`00Oomoo`0^Ool2001aOol005Aoo`03001oogoo02aoo`8007=oo`00E7oo00<0
07ooOol0:Woo0P00MGoo001DOol00`00Oomoo`0XOol2001gOol005Aoo`03001oogoo02Ioo`8007Uo
o`00E7oo00<007ooOol097oo0P00Ngoo001DOol00`00Oomoo`0ROol2001mOol005Aoo`03001oogoo
021oo`8007moo`00E7oo00<007ooOol07Woo0P00PGoo001DOol00`00Oomoo`0LOol20023Ool005Ao
o`03001oogoo01Yoo`8008Eoo`00E7oo00<007ooOol067oo0P00Qgoo001DOol00`00Oomoo`0FOol2
0029Ool005Aoo`03001oogoo01Aoo`8008]oo`00E7oo00<007ooOol04Woo0P00SGoo001DOol00`00
Oomoo`0@Ool2002?Ool005Aoo`03001oogoo00ioo`80095oo`00E7oo00<007ooOol037oo0P00Tgoo
001DOol00`00Oomoo`0:Ool2002EOol005Aoo`03001oogoo00Qoo`8009Moo`00E7oo00<007ooOol0
1Woo0P00VGoo001DOol00`00Oomoo`04Ool2002KOol005Aoo`03001oogoo009oo`8009eoo`00E7oo
00<007ooOol00P00Wgoo001DOol3002QOol005Aoo`03001oogoo0:5oo`00n7oo0000\
\>"],
  ImageRangeCache->{{{0, 247}, {81, 0}} -> {-1.10093, -0.0250053, 0.012963, 
  0.012963}}],

Cell[CellGroupData[{

Cell["Code for sketching SqrtNOTGate", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
DrawSqrtNOTGate[x_]:=
\tGraphics[{AbsoluteThickness[1],
\t          Line[{{x-1,.5},{x,.5},{x,0},{1+x,.5},{x,1},{0+x,0}}],
\t          Line[{{x+1,.5}, {x+2,.5}}],
\t          Text[\"SqrtNOT\", {x+.5,.5}, {0,0}]},
\t             AspectRatio->Automatic];
\t     \
\>", "Input",
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["Show[DrawSqrtNOTGate[0]]", "Input",
  AspectRatioFixed->True],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .33333 
MathPictureStart
%% Graphics
/Courier-Bold findfont 9  scalefont  setfont
% Scaling calculations
0.34127 0.31746 0.00793651 0.31746 [
[ 0 0 0 0 ]
[ 1 .33333 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
[ ] 0 setdash
0 g
p
P
0 0 m
1 0 L
1 .33333 L
0 .33333 L
closepath
clip
newpath
p
1 Mabswid
.02381 .16667 m
.34127 .16667 L
.34127 .00794 L
.65873 .16667 L
.34127 .3254 L
.34127 .00794 L
s
.65873 .16667 m
.97619 .16667 L
s
[(SqrtNOT)] .5 .16667 0 0 Mshowa
P
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  Evaluatable->False,
  AspectRatioFixed->True,
  ImageSize->{248, 82},
  ImageMargins->{{34, Inherited}, {Inherited, Inherited}},
  ImageCacheValid->False],

Cell[OutputFormData["\<\
Graphics[\"<<>>\"]\
\>", 
"\<\
-Graphics-\
\>"], "Output",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell["\<\
DrawNOTCircuit[x_, y_]:=
\tShow[DrawSqrtNOTGate[x], DrawSqrtNOTGate[y]]
\t     \
\>", "Input",
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["DrawNOTCircuit[0,2]", "Input",
  AspectRatioFixed->True],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .2 
MathPictureStart
%% Graphics
/Courier-Bold findfont 9  scalefont  setfont
% Scaling calculations
0.214286 0.190476 0.0047619 0.190476 [
[ 0 0 0 0 ]
[ 1 .2 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
[ ] 0 setdash
0 g
p
P
0 0 m
1 0 L
1 .2 L
0 .2 L
closepath
clip
newpath
p
p
1 Mabswid
.02381 .1 m
.21429 .1 L
.21429 .00476 L
.40476 .1 L
.21429 .19524 L
.21429 .00476 L
s
.40476 .1 m
.59524 .1 L
s
[(SqrtNOT)] .30952 .1 0 0 Mshowa
P
p
1 Mabswid
.40476 .1 m
.59524 .1 L
.59524 .00476 L
.78571 .1 L
.59524 .19524 L
.59524 .00476 L
s
.78571 .1 m
.97619 .1 L
s
[(SqrtNOT)] .69048 .1 0 0 Mshowa
P
P
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  Evaluatable->False,
  AspectRatioFixed->True,
  ImageSize->{354, 70},
  ImageMargins->{{34, Inherited}, {Inherited, Inherited}},
  ImageCacheValid->False],

Cell[OutputFormData["\<\
Graphics[\"<<>>\"]\
\>", 
"\<\
-Graphics-\
\>"], "Output",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[TextData[{
  StyleBox[
  "A SqrtNOT gate that acts on the i-th of m qubits can be defined in terms \
of creation and annihilation operators as follows (in ",
    Editable->False,
    Evaluatable->False,
    AspectRatioFixed->True],
  StyleBox["Mathematica",
    Editable->False,
    Evaluatable->False,
    AspectRatioFixed->True,
    FontSlant->"Italic"],
  StyleBox[" I is the square root of -1):",
    Editable->False,
    Evaluatable->False,
    AspectRatioFixed->True]
}], "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
SqrtNOTGate[i_, m_]:=
\tModule[{cOPim, aOPim},
\t\tcOPim = creationOP[i,m];
\t\taOPim = annihilationOP[i,m];
\t\t(1/2 (1 - I) (cOPim + aOPim) + 
\t     1/2 (1+I) (aOPim . cOPim + cOPim . aOPim)
\t    )
\t]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell["\<\
It is interesting to examine the truth table for SqrtNOTGate. \
Unlike, any classical gate which always outputs bits (0s and 1s), the \
SqrtNOTGate can return superpositions of bits.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["TruthTable[SqrtNOTGate[1,1]]", "Input",
  AspectRatioFixed->True],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {
            \(ket[0] \[Rule] 
              \((1\/2 + I\/2)\)\ ket[0] + \((1\/2 - I\/2)\)\ ket[1]\)},
          {
            \(ket[1] \[Rule] 
              \((1\/2 - I\/2)\)\ ket[0] + \((1\/2 + I\/2)\)\ ket[1]\)}
          },
        GridBaseline->{Baseline, {1, 1}},
        ColumnAlignments->{Left}],
      ColumnForm[ {ket[ 0] -> Plus[ 
          Times[ 
            Complex[ 
              Rational[ 1, 2], 
              Rational[ 1, 2]], 
            ket[ 0]], 
          Times[ 
            Complex[ 
              Rational[ 1, 2], 
              Rational[ -1, 2]], 
            ket[ 1]]], ket[ 1] -> Plus[ 
          Times[ 
            Complex[ 
              Rational[ 1, 2], 
              Rational[ -1, 2]], 
            ket[ 0]], 
          Times[ 
            Complex[ 
              Rational[ 1, 2], 
              Rational[ 1, 2]], 
            ket[ 1]]]}],
      Editable->False]], "Output"]
}, Open  ]],

Cell["\<\
Why do we call this gate a SqrtNOTGate?  To answer this look at the \
truth table of two SqrtNOT gates connected back to back:\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .2 
MathPictureStart
%% Graphics
/Courier-Bold findfont 9  scalefont  setfont
% Scaling calculations
0.214286 0.190476 0.0047619 0.190476 [
[ 0 0 0 0 ]
[ 1 .2 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
[ ] 0 setdash
0 g
p
P
0 0 m
1 0 L
1 .2 L
0 .2 L
closepath
clip
newpath
p
p
1 Mabswid
.02381 .1 m
.21429 .1 L
.21429 .00476 L
.40476 .1 L
.21429 .19524 L
.21429 .00476 L
s
.40476 .1 m
.59524 .1 L
s
[(SqrtNOT)] .30952 .1 0 0 Mshowa
P
p
1 Mabswid
.40476 .1 m
.59524 .1 L
.59524 .00476 L
.78571 .1 L
.59524 .19524 L
.59524 .00476 L
s
.78571 .1 m
.97619 .1 L
s
[(SqrtNOT)] .69048 .1 0 0 Mshowa
P
P
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  Evaluatable->False,
  AspectRatioFixed->True,
  ImageSize->{354, 70},
  ImageMargins->{{34, Inherited}, {Inherited, Inherited}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40005R0000AQ000`40O003h00OogooHgoo003oOomS
Ool004eoo`80089oo`8008moo`00CGoo00@007oo0000081oo`04001oo`00002=Ool004eoo`04001o
ogooOol2001nOol01000Oomoogoo0P00Rgoo001=Ool00`00Oomoo`03Ool2001lOol00`00Oomoo`03
Ool20029Ool004eoo`03001oogoo00Eoo`8007Yoo`03001oogoo00Eoo`8008Moo`00CGoo00<007oo
Ool01goo0P00N7oo00<007ooOol01goo0P00QGoo001=Ool00`00Oomoo`09Ool2001fOol00`00Oomo
o`09Ool20023Ool004eoo`03001oogoo00]oo`8007Aoo`03001oogoo00]oo`80085oo`00CGoo00<0
07ooOol03Goo0P00LWoo00<007ooOol03Goo0P00Ogoo001=Ool00`00Oomoo`0?Ool2001`Ool00`00
Oomoo`0?Ool2001mOol004eoo`03001oogoo015oo`8006ioo`03001oogoo015oo`8007]oo`00CGoo
00<007ooOol04goo0P00K7oo00<007ooOol04goo0P00NGoo001=Ool00`00Oomoo`0EOol2001ZOol0
0`00Oomoo`0EOol2001gOol004eoo`03001oogoo01Moo`8006Qoo`03001oogoo01Moo`8007Eoo`00
CGoo00<007ooOol06Goo0P00IWoo00<007ooOol06Goo0P00Lgoo001=Ool00`00Oomoo`0KOol2001T
Ool00`00Oomoo`0KOol2001aOol004eoo`03001oogoo01eoo`80069oo`03001oogoo01eoo`8006mo
o`00CGoo00<007ooOol07goo0P00H7oo00<007ooOol07goo0P00KGoo001=Ool00`00Oomoo`0QOol2
001NOol00`00Oomoo`0QOol2001[Ool004eoo`03001oogoo02=oo`8005aoo`03001oogoo02=oo`80
06Uoo`00CGoo00<007ooOol09Goo0P00FWoo00<007ooOol09Goo0P00Igoo001=Ool00`00Oomoo`0W
Ool2001HOol00`00Oomoo`0WOol2001UOol004eoo`03001oogoo02Uoo`8005Ioo`03001oogoo02Uo
o`8006=oo`00CGoo00<007ooOol0:goo0P00E7oo00<007ooOol0:goo0P00HGoo001=Ool00`00Oomo
o`0]Ool2001BOol00`00Oomoo`0]Ool2001OOol004eoo`03001oogoo02moo`80051oo`03001oogoo
02moo`8005eoo`00CGoo00<007ooOol0<Goo0P00CWoo00<007ooOol0<Goo0P00Fgoo001=Ool00`00
Oomoo`0cOol2001<Ool00`00Oomoo`0cOol2001IOol004eoo`03001oogoo03Eoo`8004Yoo`03001o
ogoo03Eoo`8005Moo`00CGoo00<007ooOol04Woo0`008Woo0P00B7oo00<007ooOol04Woo0`008Woo
0P00EGoo001=Ool00`00Oomoo`0COol00`00Oomoo`0SOol20016Ool00`00Oomoo`0COol00`00Oomo
o`0SOol2001COol004eoo`03001oogoo00aoo`<0009oo`<0009oo`@0009oo`@00003Ool00000009o
o`03001oo`000080009oo`<000aoo`8004Aoo`03001oogoo00aoo`<0009oo`<0009oo`@0009oo`@0
0003Ool00000009oo`03001oo`000080009oo`<000aoo`80055oo`00CGoo00<007ooOol03goo0P00
0Woo00D007ooOomoo`0000Aoo`03001oogoo00=oo`03001oo`00008000=oo`04001oogoo000?Ool2
0012Ool00`00Oomoo`0?Ool20002Ool01@00Oomoogoo000017oo00<007ooOol00goo00<007oo0000
0P000goo00@007ooOol000moo`8004moo`0037oo@P0047oo0`000Woo00D007ooOomoo`0000Aoo`03
001oogoo00=oo`03001oo`00008000=oo`04001oogoo000@Oom4000@Ool30002Ool01@00Oomoogoo
000017oo00<007ooOol00goo00<007oo00000P000goo00@007ooOol0011ood<000eoo`00CGoo00<0
07ooOol037oo0P000Woo00@007ooOol000=oo`03001oogoo009oo`03001oogoo00=oo`800003Ool0
000000=oo`04001oogoo000>Ool20013Ool00`00Oomoo`0<Ool20002Ool01000Oomoo`000goo00<0
07ooOol00Woo00<007ooOol00goo0P0000=oo`0000000goo00@007ooOol000ioo`80051oo`00CGoo
00<007ooOol037oo00<007ooOol00Woo0`000Woo2@000goo0P0000=oo`0000000goo0P0000Aoo`00
Ool000Yoo`8004Eoo`03001oogoo00aoo`03001oogoo009oo`<0009oo`T000=oo`800003Ool00000
00=oo`800004Ool007oo000:Ool2001BOol004eoo`03001oogoo00eoo`<000aoo`03001oogoo009o
o`80009oo`D00003Ool0000000<000Qoo`8004Moo`03001oogoo00eoo`<000aoo`03001oogoo009o
o`80009oo`D00003Ool0000000<000Qoo`8005Aoo`00CGoo00<007ooOol077oo00<007ooOol05goo
0P00BGoo00<007ooOol077oo00<007ooOol05goo0P00EWoo001=Ool00`00Oomoo`0dOol2001;Ool0
0`00Oomoo`0dOol2001HOol004eoo`03001oogoo039oo`8004eoo`03001oogoo039oo`8005Yoo`00
CGoo00<007ooOol0<7oo0P00Cgoo00<007ooOol0<7oo0P00G7oo001=Ool00`00Oomoo`0^Ool2001A
Ool00`00Oomoo`0^Ool2001NOol004eoo`03001oogoo02aoo`8005=oo`03001oogoo02aoo`80061o
o`00CGoo00<007ooOol0:Woo0P00EGoo00<007ooOol0:Woo0P00HWoo001=Ool00`00Oomoo`0XOol2
001GOol00`00Oomoo`0XOol2001TOol004eoo`03001oogoo02Ioo`8005Uoo`03001oogoo02Ioo`80
06Ioo`00CGoo00<007ooOol097oo0P00Fgoo00<007ooOol097oo0P00J7oo001=Ool00`00Oomoo`0R
Ool2001MOol00`00Oomoo`0ROol2001ZOol004eoo`03001oogoo021oo`8005moo`03001oogoo021o
o`8006aoo`00CGoo00<007ooOol07Woo0P00HGoo00<007ooOol07Woo0P00KWoo001=Ool00`00Oomo
o`0LOol2001SOol00`00Oomoo`0LOol2001`Ool004eoo`03001oogoo01Yoo`8006Eoo`03001oogoo
01Yoo`80079oo`00CGoo00<007ooOol067oo0P00Igoo00<007ooOol067oo0P00M7oo001=Ool00`00
Oomoo`0FOol2001YOol00`00Oomoo`0FOol2001fOol004eoo`03001oogoo01Aoo`8006]oo`03001o
ogoo01Aoo`8007Qoo`00CGoo00<007ooOol04Woo0P00KGoo00<007ooOol04Woo0P00NWoo001=Ool0
0`00Oomoo`0@Ool2001_Ool00`00Oomoo`0@Ool2001lOol004eoo`03001oogoo00ioo`80075oo`03
001oogoo00ioo`8007ioo`00CGoo00<007ooOol037oo0P00Lgoo00<007ooOol037oo0P00P7oo001=
Ool00`00Oomoo`0:Ool2001eOol00`00Oomoo`0:Ool20022Ool004eoo`03001oogoo00Qoo`8007Mo
o`03001oogoo00Qoo`8008Aoo`00CGoo00<007ooOol01Woo0P00NGoo00<007ooOol01Woo0P00QWoo
001=Ool00`00Oomoo`04Ool2001kOol00`00Oomoo`04Ool20028Ool004eoo`03001oogoo009oo`80
07eoo`03001oogoo009oo`8008Yoo`00CGoo00<007ooOol00P00Ogoo00<007ooOol00P00S7oo001=
Ool30021Ool3002>Ool004eoo`03001oogoo085oo`03001oogoo08ioo`00ogooHgoo0000\
\>"],
  ImageRangeCache->{{{0, 353}, {69, 0}} -> {-1.1859, -0.0250052, 0.0152176, 
  0.0152176}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["TruthTable[SqrtNOTGate[1,1] . SqrtNOTGate[1,1]]", "Input",
  AspectRatioFixed->True],

Cell["\<\
The net result of two SqrtNOTGates connected back to back, is a \
unitary operation (in fact the NOT operation). \
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["UnitaryQ[SqrtNOTGate[1,1] . SqrtNOTGate[1,1]]", "Input",
  AspectRatioFixed->True],

Cell["UnitaryQ[NOT]", "Input",
  AspectRatioFixed->True]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Representing the Computation as a Circuit", "Section",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
In general , the quantum memory register of a Feynman-like quantum \
computer will consist of a set of \"cursor\" qubits (which keep track of the \
progress of the computation) and a set of \"program\" qubits through which \
the \"input\" is fed into the computer and the \"output\" is extracted (when \
it becomes available). We want to apply specific operators, corresponding to \
the action of logic gates, on just the \"program\" qubits.  Hence we must \
specify which qubits a given operator must act upon.  In the case of the \
simple SqrtNOTGate squared circuit, we will use two SqrtNOTGates that both \
act on the 4th of 4 qubits.  Thus the NOT circuit, built from two \
SqrtNOTGates connected back to back, is specified as follows:\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["sqrtNOTcircuit = {SqrtNOTGate[4,4], SqrtNOTGate[4,4]};", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell["\<\
This contains an ordered list of quantum gates together with the \
input \"lines\" these gates work upon.

The embedded NOT gate (NOTGate[i,m]), the embedded CONTROLLED-NOT gate \
(CNGate[i,j,m]) and the embedded CONTROLLED-CONTROLLED-NOT gate \
(CCN[i,j,k,m]) are defined similarly. The arguments i, j, k label the lines \
of the gates and m signifies the total number of lines in the circuit. Thus, \
a controlled-NOT gate acting on the 2nd and 4th of 5 lines would be \
CNGate[2,4,5] etc.

You can compare the square root of NOT gate and its embedded form easily.  A \
square root of NOT gate acting on the 1st of 1 qubit (i.e. an unembedded \
SqrtNOTGate) is given by sqrtNOT11 in the example below:\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["\<\
SetOptions[$Output, PageWidth->Infinity];

sqrtNOT11 = SqrtNOTGate[1,1];  (* unembedded gate *)

MatrixForm[sqrtNOT11, 
           TableSpacing->{0,4}, TableAlignments->{Center, Center}]\
\>", 
  "Input",
  AspectRatioFixed->True],

Cell[BoxData[
    TagBox[
      RowBox[{"(", GridBox[{
            {\(1\/2 + I\/2\), \(1\/2 - I\/2\)},
            {\(1\/2 - I\/2\), \(1\/2 + I\/2\)}
            }], ")"}],
      (MatrixForm[ #, TableSpacing -> {0, 4}, TableAlignments -> {Center, 
        Center}]&)]], "Output"]
}, Open  ]],

Cell["\<\
The square root of NOT gate that acts on the 2nd of 3 qubits is \
given by sqrtNOT23 in the next example:\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["\<\
SetOptions[$Output, PageWidth->Infinity];

sqrtNOT23 = SqrtNOTGate[2,3];

MatrixForm[sqrtNOT23, 
           TableSpacing->{0,4}, TableAlignments->{Center, Center}]\
\>", 
  "Input",
  AspectRatioFixed->True],

Cell[BoxData[
    TagBox[
      RowBox[{"(", GridBox[{
            {\(1\/2 + I\/2\), "0", \(1\/2 - I\/2\), "0", "0", "0", "0", "0"},
            {"0", \(1\/2 + I\/2\), "0", \(1\/2 - I\/2\), "0", "0", "0", "0"},
            {\(1\/2 - I\/2\), "0", \(1\/2 + I\/2\), "0", "0", "0", "0", "0"},
            {"0", \(1\/2 - I\/2\), "0", \(1\/2 + I\/2\), "0", "0", "0", "0"},
            {"0", "0", "0", "0", \(1\/2 + I\/2\), "0", \(1\/2 - I\/2\), "0"},
            {"0", "0", "0", "0", "0", \(1\/2 + I\/2\), "0", \(1\/2 - I\/2\)},
            {"0", "0", "0", "0", \(1\/2 - I\/2\), "0", \(1\/2 + I\/2\), "0"},
            {"0", "0", "0", "0", "0", \(1\/2 - I\/2\), "0", \(1\/2 + I\/2\)}
            }], ")"}],
      (MatrixForm[ #, TableSpacing -> {0, 4}, TableAlignments -> {Center, 
        Center}]&)]], "Output"]
}, Open  ]],

Cell["\<\
Other embedded gates may be derived in a similar fashion.

Notice that the embedded gates are still unitary:\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["UnitaryQ[sqrtNOT23]", "Input",
  AspectRatioFixed->True]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Determining the Size of the Memory Register", "Section",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
For the NOT circuit, built from two SqrtNOTGates connected back to \
back, there are 2 gates (k=2), so we need need k+1=3 cursor qubits (to track \
the progress of the computation) and 1 input/output qubit (m=1) which will \
serve a dual purpose; to enter the input to the circuit and to record the \
output. Hence we need 4 qubits in all; 3 cursor qubits plus one program \
qubit, making a total of m+k+1=4 qubits for the entire quantum memory \
register.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Computing the Hamiltonian Operator", "Section",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
In Feynman's quantum computer, the Hamiltonian is time independent \
and consists of a sum of terms describing the advance and retreat of the \
computation.  The net effect of the Hamiltonian is to place the memory \
register of the Feynman quantum computer in a superposition of states \
representing the same computation at various stages of completion. The \
command for generating the Hamiltonian of a Feynman quantum computer is \
\"Hamiltonian\".\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["?Hamiltonian", "Input",
  AspectRatioFixed->True],

Cell[BoxData[
    \("Hamiltonian[m, k, circuit] returns the time independent  Hamiltonian \
matrix corresponding to the given circuit.  The circuit consists of k quantum \
gates and a total of  m lines (i.e. m inputs and m outputs)."\)], "Print"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
SetOptions[$Output, PageWidth->Infinity];
H = Hamiltonian[1, 2, sqrtNOTcircuit];
MatrixForm[H, TableSpacing->{0,4}, TableAlignments->{Center,Center}]\
\>", 
  "Input",
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Code", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
Hamiltonian[m_, k_, circuit_]:=
\tModule[{terms},
\t\tterms = Table[term[i, m+k+1, circuit[[i]]], {i, 1, k}];
\t\tApply[Plus, terms + Map[Conjugate[Transpose[#]]&, terms]]
\t]

Hamiltonian::usage =
\t\"Hamiltonian[m, k, circuit] returns the time independent \\
\tHamiltonian matrix corresponding to the given circuit. \\
\tThe circuit consists of k quantum gates and a total of \\
\tm lines (i.e. m inputs and m outputs).\";
\t
term[i_, l_, gate_]:=
\tcreationOP[i+1, l] . annihilationOP[i, l] . gate\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Computing the Unitary Evolution Operator", "Section",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
The unitary evolution operator is derived from the solution to \
Schrodinger's equation for the evolution of the memory register i.e. | psi(t) \
> = U(t) | psi(0) > where U(t) = e^(-i H t / hBar) where H is the \
(time-independent) Hamiltonian, i is square root of -1, hBar is Planck's \
constant over 2 Pi and t is the time. Remember, U(t) and H are really square \
matrices so the necessary exponential is a matrix exponential.  We set hBar=1 \
for simplicity.   In terms of the code, we call U(t), for a particular \
circuit at a particular time, EvolutionOP[m,k,circuit,t].\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["?EvolutionOP", "Input",
  AspectRatioFixed->True],

Cell[BoxData[
    \("EvolutionOP[m, k, circuit, t] is the time dependent evolution  \
operator for the given circuit. The circuit consists of k  quantum gates and \
a total of m lines (i.e. m inputs and m  outputs)."\)], "Print"]
}, Open  ]],

Cell["\<\
U is given by the t-th matrix power of expH i.e. U = \
MatrixPower[expH, t] where expH is the matrix exponential of H.  You may need \
to scroll sideways to see the structure of expH.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["Code", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
EvolutionOP[m_, k_, circuit_, t_ ]:=
\tMatrixPower[N[MatrixExp[ -I Hamiltonian[m, k, circuit]]], t]
\t
EvolutionOP::usage =
\t\"EvolutionOP[m, k, circuit, t] is the time dependent evolution \\
\toperator for the given circuit. The circuit consists of k \\
\tquantum gates and a total of m lines (i.e. m inputs and m \\
\toutputs).\";\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
SetOptions[$Output, PageWidth->Infinity];
expH = MatrixExp[ -I Hamiltonian[1, 2, sqrtNOTcircuit]];
MatrixForm[expH, TableSpacing->{0,0}]\
\>", "Input",
  AspectRatioFixed->True]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Running the Quantum Computer for a Fixed Length of Time", "Section",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
To simulate running the quantum computer, that implements a given \
circuit, for a specific length of time, use the function \
SchrodingerEvolution.  This function takes 5 arguments: the initial state of \
the memory register, the number of inputs/outputs in the circuit, the number \
of gates in the circuit, the circuit matrix expressed as a product of \
embedded gates and the duration of the simulation.  The result is the state \
of the memory register at the end of the simulation period.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["?SchrodingerEvolution", "Input",
  AspectRatioFixed->True],

Cell[BoxData[
    \("SchrodingerEvolution[initKet, m, k, circuit, t]  evolves the given \
circuit for time t from the initial configuration  initKet (a ket vector e.g. \
ket[1,0,0,0]). You need to specify that  there are k gates in the circuit (so \
there are k+1 cursor bits) and  that there are m program bits (i.e. the \
number of bits used as input  data not counting the cursor bits)."\)], "Print"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Code", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
SchrodingerEvolution[ket_, m_, k_, circuit_, t_]:=
\tColumnVectorToKet[Chop[EvolutionOP[m,k,circuit,t] . 
\t                       KetToColumnVector[ ket ]
\t                      ]
\t                 ]
SchrodingerEvolution::usage =
\t\"SchrodingerEvolution[initKet, m, k, circuit, t] \\
\tevolves the given circuit for time t from the initial configuration \\
\tinitKet (a ket vector e.g. ket[1,0,0,0]). You need to specify that \\
\tthere are k gates in the circuit (so there are k+1 cursor bits) and \\
\tthat there are m program bits (i.e. the number of bits used as input \\
\tdata not counting the cursor bits).\";\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["\<\
sqrtNOTcircuit = {SqrtNOTGate[4,4], SqrtNOTGate[4,4]};
SchrodingerEvolution[ket[1,0,0,0], 1, 2, sqrtNOTcircuit, 0.5]\
\>", "Input",
  AspectRatioFixed->True],

Cell[OutputFormData["\<\
-0.1198777014621846*ket[0, 0, 1, 1] + 
 
  (0.229681342466392 - 0.2296813424663922*I)*ket[0, 1, 0, 0] + 
 
  (-0.2296813424663921 - 0.2296813424663922*I)*ket[0, 1, 0, 1] + 
 
  0.880122298537815*ket[1, 0, 0, 0]\
\>", 
"\<\
-0.119878 ket[0, 0, 1, 1] + (0.229681 - 0.229681 I) ket[0, 1, 0, 0] + 
 
  (-0.229681 - 0.229681 I) ket[0, 1, 0, 1] + 0.880122 ket[1, 0, 0, 0]\
\>"], 
  "Output",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell["Is this a properly normalized ket?", "Text",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["NormalizedKetQ[%]", "Input",
  AspectRatioFixed->True],

Cell[OutputFormData["\<\
True\
\>", "\<\
True\
\>"], "Output",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell["\<\
Do you get the same answer is you run the simulator again, on the \
same input, for the same length of time?\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
sqrtNOTcircuit = {SqrtNOTGate[4,4], SqrtNOTGate[4,4]};
SchrodingerEvolution[ket[1,0,0,0], 1, 2, sqrtNOTcircuit, 0.5]\
\>", "Input",
  AspectRatioFixed->True],

Cell["Were you surprised by the outcome?", "Text",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Explanation in here ...", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
The Schrodinger equation is completely deterministic. Thus if you \
evolve the same circuit, with the same input, for the same length of time, \
the outcome, ignoring errors, will be identical each time. It is only when \
you make a measurement that randomness enters the picture.
        Be aware that some physicists would contest the last statement.  \
Outcomes of measurements are random according to the \"Copenhagen\" \
interpretation of quantum mechanics.  Other interpretations, such as \
Everitt's \"Many-Worlds\" interpretation and Cerf & Adami's interpretation, \
explain measurement differently.  However, all interpretations accept that \
the Schrodinger equation is a deterministic differential equation.\
\>", 
  "Text",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Running the Quantum Computer Until the Computation is Done", "Section",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
In a Feynman-like quantum computer, the position of the cursor \
keeps track of the logical progress of the computation. If the computation \
can be accomplished in k+1 (logic gate) operations, the cursor will consist \
of a chain of k+1 atoms, only one of which can ever be in the |1> state. The \
cursor keeps track of how many,\tlogical operations have been applied to the \
program bits thus far. Thus if you measure the cursor position and find it at \
the third site, say, then you know that the memory register will, at that \
moment, contain the result of applying the first three gate operations to the \
input state. This does not mean that only three such operations have been \
applied. In the Feynman computer the computation proceeds forwards and \
backwards simultaneously. As time progresses, the probability of finding the \
cursor at the (k+1)-th site rises and falls. If you are lucky, and happen to \
measure the cursor position when the probability of the cursor being at the \
(k+1)-th site is high, then you have a good chance of finding it there.

Operationally, you periodically measure the cursor position. This collapses \
the superposition of states that represent the cursor position but leaves the \
superposition of states in the program bits unscathed. If the cursor is not \
at the (k+1)-th site then you allow the computer to evolve again from the \
new, (partially) collapsed state. However, as soon as the cursor is found at \
the (k+1)-th site, the computation is halted and the complete state of the \
memory register (cursor bits and program bits) is measured. Whenever the \
cursor is at the (k+1)-th site, a measurement of the state of the program \
bits at that moment is guaranteed to return a valid answer to the computation \
the quantum computer was working on. So in the Feynman model of a quantum \
computer, there is no doubt at to the correctness of the answer, merely the \
time at which the answer is available.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
To run the quantum computer until completion, you must periodically \
check the cursor position to see if all the gate operations have been \
applied. As soon as you find the cursor in its extreme position, you can be \
sure that, at that moment, a correct answer is obtainable from reading the \
program qubits. Note that we say \"a\" correct answer and not \"the\" correct \
answer because, if a problem admits more than one acceptable solution, then \
the final state of the  Feynman's quantum computer will contain a \
superposition of all the valid answers.  Upon measurement only one of these \
answers will be obtained however.

To read just the cursor, one simply restricts measurements of the memory \
register to be made on just those qubits used to keep track of the cursor.  \
Thus the program qubits (that contain the answer) are NOT measured in this \
process.  Nevertheless, the outcome of the measurement of the cursor position \
causes the relative state of the program qubits to be projected into a \
subspace that is consistent with the position of the cursor that is found.   \
For example, if the cursor position indicates that, say, the first N gate \
operations have been applied, then the program qubits are projected into a \
superposition corresponding to the state created by applying just the first N \
gate operations in the circuit.  This idea of measurements of one part of a \
memory register affecting the relative state of the other (unmeasured) part \
of the same register, is crucial to understanding the operation of quantum \
computers.  In general you do not want to make any measurements on the \
program qubits (the qubits which contain the answer) until you can be sure \
that an answer is available. The command for reading the cursor position is \
ReadCursorBits:\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["?ReadCursorBits", "Input",
  AspectRatioFixed->True],

Cell[BoxData[
    \("ReadCursorBits[numCursorBits, superposition] reads the state  of the \
cursor bits of a Feynman-like quantum computer. If the  computation can be \
accomplished in k+1 (logic gate) operations,  the cursor will consist of a \
chain of k+1 atoms, only one of which  can ever be in the |1> state. The \
cursor keeps track of how many,  logical operations have been applied to the \
program bits thus far.  The state of the program bits of the computer are \
unaffected by  measuring just the cursor bits. If the cursor is ever found at \
the  (k+1)-th site, then, if you measured the program bits at that moment,  \
they would be guaranteed to contain a valid answer to the computation  the \
quantum computer was working on."\)], "Print"]
}, Open  ]],

Cell["\<\
For our sqrtNOTcircuit, we have 2 gates (i.e. k=2) and therefore \
k+1=3 cursor positions.  The initial state of the cursor is |100> and the \
input to the circuit is set to |0>.  Thus the overall memory register is \
initially in the state |1000> (which we represent as ket[1,0,0,0]).  To read \
the cursor position after a time t=1.1 has elapsed use ReadCursorBits[3, \
evoln1] where evoln1 is the state of the entire memory register after the \
computer has evolved for time t=1.1.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["Code", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
Options[ReadCursorBits] = {TraceProgress->False};

ReadCursorBits[numBitsToRead_, w_. ket[bits__] + kets_., opts___]:=
\tModule[{nBits, superposition, resultsPerStep, traceQ},
\t\ttraceQ = TraceProgress /. {opts} /. Options[ReadCursorBits];
\t\tnBits = Length[{bits}]; (* figure out number of bits in memory *)
\t\tsuperposition = {\"BeforeAnyMeasurements\", w ket[bits] + kets};
\t\tresultsPerStep = FoldList[MeasureIthBit[#2,#1,nBits]&,
\t\t                          superposition,
\t\t                          Range[numBitsToRead]
\t\t                         ];
\t\tWhich[traceQ===False,
\t\t        {Rest[Map[#[[1]]&, resultsPerStep]],  (* results for bits *)
\t\t         Last[resultsPerStep][[2]]            (* projected state *)
\t\t        },
\t\t      traceQ===True,
\t\t        ColumnForm[resultsPerStep]
\t\t     ]
\t]/;(numBitsToRead <= Length[{bits}])

ReadCursorBits::usage =
\t\"ReadCursorBits[numCursorBits, superposition] reads the state \\
\tof the cursor bits of a Feynman-like quantum computer. If the \\
\tcomputation can be accomplished in k+1 (logic gate) operations, \\
\tthe cursor will consist of a chain of k+1 atoms, only one of which \\
\tcan ever be in the |1> state. The cursor keeps track of how many, \\
\tlogical operations have been applied to the program bits thus far. \\
\tThe state of the program bits of the computer are unaffected by \\
\tmeasuring just the cursor bits. If the cursor is ever found at the \\
\t(k+1)-th site, then, if you measured the program bits at that moment, \\
\tthey would be guaranteed to contain a valid answer to the computation \\
\tthe quantum computer was working on.\";\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
sqrtNOTcircuit = {SqrtNOTGate[4,4], SqrtNOTGate[4,4]};
state1  = SchrodingerEvolution[ket[1,0,0,0],1,2,sqrtNOTcircuit,1.1];

{cursor1, projected1} = ReadCursorBits[3, state1]\
\>", "Input",
  AspectRatioFixed->True],

Cell["\<\
The output consists of a list, {cursor-position, projected-state}, \
showing the cursor position and the state of the entire memory register after \
the cursor position has been measured.  For example, the output:\
\>", "Text",\

  Evaluatable->False,
  AspectRatioFixed->True],

Cell[OutputFormData[
"\<\
{{0, 1, 0}, (0.5 - 0.5000000000000002*I)*ket[0, 1, 0, 0] + 
 
   (-0.5 - 0.5000000000000001*I)*ket[0, 1, 0, 1]}\
\>", 
"\<\
{{0, 1, 0}, (0.5 - 0.5 I) ket[0, 1, 0, 0] + (-0.5 - 0.5 I) ket[0, 1, 0, \
1]}\
\>"], "Output",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
would indicate that the cursor is in the second position ({0,1,0}), \
and the state of the entire memory register is then  (0.5 - 0.5 i) |0100> + \
(-0.5 - 0.5 i) |0101> (where i is the square root of -1).  You can compare \
the state of the register after measuring the cursor position (projected1) to \
its state before measuring the cursor position (state1) by simply asking for \
the values of these variables.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["Compare the following two states:", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["projected1 ", "Input",
  AspectRatioFixed->True],

Cell["state1", "Input",
  AspectRatioFixed->True],

Cell["\<\
If you measure the cursor position of two identically prepared \
quantum computers at identical times would you find both cursors at the same \
place?\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
The initial set up is identical, is the result of reading the \
cursor?\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
sqrtNOTcircuit = {SqrtNOTGate[4,4], SqrtNOTGate[4,4]};
state2 = SchrodingerEvolution[ket[1,0,0,0],1,2,sqrtNOTcircuit,1.1];

{cursor2, projected2} = ReadCursorBits[3, state2]\
\>", "Input",
  AspectRatioFixed->True],

Cell["Are the cursors found at the same position?", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["cursor1 === cursor2", "Input",
  AspectRatioFixed->True],

Cell["\<\
Are the states of the memory registers the same after measuring the \
cursors?\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["projected1 === projected2", "Input",
  AspectRatioFixed->True],

Cell["\<\
Were the states of the memory registers the same before the cursor \
positions were measured?\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["state1 === state2", "Input",
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Explanation in here ...", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
In general, you will obtain different results for the cursor \
position in the two cases, although by chance you might have obtained the \
same answer both times!  If so, try running the last experiment again a few \
times.

If two identically prepared quantum computers are allowed to evolve for \
identical times, then, as the Schrodinger equation is deterministic, the two \
memory registers will evolve into identical superpositions  (ignoring errors \
of course).  Hence state1 will always be identical to state2.  However, when \
you measure the cursor positions, you cause each of the superpositions to \
\"collapse\" in a random way independent of one another.  In one case you may \
find, say, 2 gate operations have been applied, and in the other you may find \
say 1 gate operation has been applied.  Thus cursor1 is not, in general, the \
same as cursor2.   Consequently,  the relative states of the program qubits \
of each computer will then also be different after the measurements of the \
cursor.  Hence projected1 is different from projected2, in general.\
\>", 
  "Text",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Reading the Memory Register (Cursor Qubits & Program Qubits)", 
  "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
Once the cursor is found at a position indicating that all of the \
gate operations have been applied, i.e. at its k+1-th site, then you can \
extract an answer from the computer by reading all the qubits in the memory \
register.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["?ReadMemoryRegister", "Input",
  AspectRatioFixed->True],

Cell[BoxData[
    \("ReadMemoryRegister[superposition] reads the state of each bit  in the \
memory register. As the i-th and j-th bit measurement  operators commute (for \
any i and j), it does not matter in what  order you measure the bits."\)], 
  "Print"]
}, Open  ]],

Cell["\<\
If a computation has only one answer, you will always obtain the \
single correct answer.  If a computation admits many possible answers, you \
are equally likely to obtain any one of them upon measuring the final state \
of the register.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["Code", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
(*====================*)
(* ReadMemoryRegister *)
(*====================*)
(* Given a superposition representing the state of the memory of a
   quantum computer, return the result of measuring the memory.
*)
Options[ReadMemoryRegister] = {TraceProgress->False};

ReadMemoryRegister[w_. ket[bits__] + kets_., opts___]:=
\tModule[{nBits, superposition, resultsPerStep, traceQ},
\t\ttraceQ = TraceProgress /. {opts} /. Options[ReadMemoryRegister];
\t\tnBits = Length[{bits}]; (* figure out number of bits in memory *)
\t\tsuperposition = {\"BeforeAnyMeasurements\", w ket[bits] + kets};
\t\tresultsPerStep = FoldList[MeasureIthBit[#2,#1,nBits]&,
\t\t                          superposition,
\t\t                          Range[nBits]
\t\t                         ];
\t\tWhich[traceQ===False, 
\t\t        {Rest[Map[#[[1]]&, resultsPerStep]],  (* results for bits *)
\t\t         Last[resultsPerStep][[2]]            (* projected state *)
\t\t        },
\t\t      traceQ===True,
\t\t        ColumnForm[resultsPerStep] (*list of {results,projectedStates}*)
\t\t     ]
\t]
\t
ReadMemoryRegister::usage =
\t\"ReadMemoryRegister[superposition] reads the state of each bit \\
\tin the memory register. As the i-th and j-th bit measurement \\
\toperators commute (for any i and j), it does not matter in what \\
\torder you measure the bits.\";
\t\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell["\<\
MeasureIthBit[i_, {_, superposition_}, nBits_]:=
\tModule[{p1, zeroOrOne, projectedState},
\t\tp1 = ProbabilityIthBitIs1[i, superposition];
\t\tzeroOrOne = BiasedSelect[{0,1}, {1-p1, p1}];
\t\tprojectedState = 
\t\t  SuperpositionWithIthBitFixed[i, zeroOrOne, nBits, superposition];
\t\t{zeroOrOne, projectedState}
\t]
\t\t       
KetWithIthBitZeroOrOne[i_, zeroOrOne_, nBits_]:=
\tReplacePart[Apply[ket, Table[_,{nBits}]], zeroOrOne, i]

SuperpositionWithIthBitFixed[_, _, _, w_. ket[bits__]]:=
  NormalizeKet[w ket[bits]]
SuperpositionWithIthBitFixed[i_, zeroOrOne_, nBits_, superposition_]:=
  NormalizeKet[Select[superposition,
\t\t              MatchQ[#, _. KetWithIthBitZeroOrOne[i,zeroOrOne,nBits]
\t\t                    ]&
\t\t             ]
\t\t      ]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell["\<\
ProbabilityIthBitIs1[i_, w_. ket[bits__]]:=
\tIf[ket[bits][[i]] == 1, Abs[w]^2, 1-Abs[w]^2]  (* Abs[w]^2 == 1 or 0 only \
*)
\t
ProbabilityIthBitIs1[i_, w_. ket[bits__] + kets_.]:=
\tModule[{nBits, terms},
\t\tnBits = Length[{bits}];
\t\tterms = Select[w ket[bits] + kets, 
\t\t               MatchQ[#, _. KetWithIthBit1[i,nBits]]&];
\t\tN[Apply[Plus, Map[Abs[#]^2&, Amplitudes[terms]]]/
\t\t  Apply[Plus, Map[Abs[#]^2&, Amplitudes[w ket[bits] + kets]]]
\t\t ]
\t]
\t
ProbabilityIthBitIs1[i_, c_. (w_. ket[bits__] + kets_.)]:=
\tProbabilityIthBitIs1[i, Expand[c (w ket[bits] + kets)]]

ProbabilityIthBitIs1::usage =
\t\"The state of the memory register of a quantum computer \\
\t(that is comprised of m 2-state particles) is represented by a \\
\tsuperposition 2^m eigenstates. The function \\
\tProbabilityIthBitIs1[i, superposition] computes the probability \\
\tthat, upon being measured, the i-th, of the m, bits will be a 1.\";
\t
KetWithIthBit1[i_, nBits_]:=
\tReplacePart[Apply[ket, Table[_,{nBits}]], 1, i]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell["\<\
(* The list of probabilities should sum to 1. The call to Partition \

   constructs a set of probability intervals whose width is proportional
   to the probability with which the corresponding element in list 
   is selected.
*)
BiasedSelect[list_, probabilities_]:=
\tModule[{random},
\t\trandom  = Random[];
\t\tApply[Part[list, #]&,
\t\t      Flatten[
\t\t       Position[Map[InRangeQ[random, #]&, 
\t\t                    Partition[FoldList[Plus,0,probabilities],2,1]
\t\t                   ],
\t\t                True
\t\t       ]
\t\t      ]
\t\t     ]
\t] (* /;CheckProbabilitiesQ[probabilities] *)

BiasedSelect::usage =
\t\"BiasedSelect[{e1,e2,...,en}, {p1,p2,...,pn}] returns element ei of \\
\tthe first list with probability given in the second list pi.\";

BiasedSelect::probabilityLeak =
\t\"You have a probability leak. The probabilities you specified do \\
\tnot add up to 1.\";

BiasedSelect::excess =
\t\"The probabilities you specified sum to greater than 1.\";

CheckProbabilitiesQ[probabilities_]:=
\tModule[{psum = Apply[Plus, probabilities]},
\t\tWhich[psum<1,  Message[BiasedSelect::probabilityLeak],
\t\t      psum>1,  Message[BiasedSelect::excess],
\t\t      psum==1, True
\t\t     ]
\t]
\t
InRangeQ[n_, {lb_, 1}]:=   lb <= n <= 1
InRangeQ[n_, {lb_, ub_}]:= lb <= n < ub\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Evolving the Quantum Computer Until Complete", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
The above steps are bundled together in the command EvolveQC.  \
EvolveQC allows you to evolve the quantum computer, checking whether an \
answer is available and, if so,  measuring the memory register to extract it. \
 The output from EvolveQC is a complete history of the evolution of the \
quantum computer.  This history consists of a sequence of 4-element \
snapshots.  Each snapshot shows the time at which the cursor was measured, \
the state of the register before the cursor was measured, the result of the \
measurment and the state into which the register is projected because of the \
measurement.  You can use the option TimeBetweenObservations to control the \
time between observations of the cursor and whether the intervals should be \
regular or random. The default interval is 1 time unit.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["?EvolveQC", "Input",
  AspectRatioFixed->True],

Cell[BoxData[
    \("The function EvolveQC[initState, circuit] evolves a Feynman-like  \
quantum computer, specified as a circuit of interconnected  quantum logic \
gates, from some initial state until the  computation is complete. The output \
is a list of snapshots of the  state of the QC at successive \
cursor-measurement times. Each  snapshot consists of a 4 element list whose \
elements are the  time at which the cursor is measured, the state of the \
register  immediately before the cursor is measured, the result of the  \
measurement and the state of the register immediately after  the cursor \
position is measured. The latter is the projected  state of the register. \
EvolveQC can take the optional argument  TimeBetweenObservations which can be \
set to a number or a  probability distribution. The default time between \
observations  is 1 time unit."\)], "Print"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Code", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
Needs[\"Statistics`ContinuousDistributions`\"]  (* for \
distributions *)
Needs[\"Statistics`DiscreteDistributions`\"]

Options[EvolveQC] = {TimeBetweenObservations->1, Explain->False};
                    
EvolveQC[w_. ket[bits__] + kets_., circuit_, opts___]:=
\tModule[{m, k, cursor, tau, explain, state, history},
\t\tk = Length[circuit];   (* the circuit consists of k operators *)
\t\tm = Length[{bits}]-(k+1);
\t\tcursor = Join[{1}, Table[0,{k}]]; (* initial state of cursor *)
\t\ttau = TimeBetweenObservations /. {opts} /. Options[EvolveQC];
\t\texplain = Explain /. {opts} /. Options[EvolveQC];
\t\t
\t\tstate = {0, w ket[bits] + kets, cursor, w ket[bits] + kets};
\t\thistory = {state};
\t\t
\t\tIf[explain===True, ExplainResult[state]];
\t\tWhile[Not[ComputationCompleteQ[state]] && CursorOkQ[state],
\t\t\t      (state = EvolveForTimeTau[state,m,k,N[circuit],EvolnTime[tau]];
\t\t\t       If[explain===True, ExplainResult[state]];
\t\t\t       AppendTo[history, state]
\t\t\t      )
\t\t\t     ];
\t\t
\t\tWhich[Not[CursorOkQ[state]],  
\t\t              (Message[EvolveQC::badCursor]; 
\t\t               AppendTo[history, 
\t\t                        ReplacePart[state, $BadCursor, 3]
\t\t                       ]
\t\t              ), (* return history with bad cursor flag *)
\t\t              
\t\t       explain===False,       history,    (* return complete history *)
\t\t       explain===True,        Print[\"You can now read an answer from the \
memory register that is guaranteed to be correct!\"]
\t\t     ]
\t]
\t
EvolveQC::badCursor = 
\t\"The cursor was corrupted. No useful answers may be extracted \\
\tfrom the computer.\";

EvolveQC::usage =
\t\"The function EvolveQC[initState, circuit] evolves a Feynman-like \\
\tquantum computer, specified as a circuit of interconnected \\
\tquantum logic gates, from some initial state until the \\
\tcomputation is complete. The output is a list of snapshots of the \\
\tstate of the QC at successive cursor-measurement times. Each \\
\tsnapshot consists of a 4 element list whose elements are the \\
\ttime at which the cursor is measured, the state of the register \\
\timmediately before the cursor is measured, the result of the \\
\tmeasurement and the state of the register immediately after \\
\tthe cursor position is measured. The latter is the projected \\
\tstate of the register. EvolveQC can take the optional argument \\
\tTimeBetweenObservations which can be set to a number or a \\
\tprobability distribution. The default time between observations \\
\tis 1 time unit.\";

(* Cursor is ok so long as it contains exactly 1 bit *)
CursorOkQ[{_,_,cursor_,_}]:=
\tIf[Count[cursor,1] == 1, True, False]

(* Computation is complete if the last cursor bit is a 1.
   The related logical test CursorOkQ, will abort the While loop
   if the cursor is ever corrupted (i.e. contains no bits or more than
   one bit). Hence the simple termination test, of checking the
   last bit, is sufficient.
*)
ComputationCompleteQ[{_, _, {___,1}, _}]:=True
ComputationCompleteQ[_]:=False

EvolnTime[time_Integer]:= time
EvolnTime[time_Real]:= time
EvolnTime[dist_[parameters___]]:=
\tIf[StringMatchQ[ToString[dist], \"*Distribution\"],
\t   Random[dist[parameters]],
\t   Message[EvolnTime::notknown]; Abort[]]
EvolnTime::notknown = 
\t\"The time between observations of the cursor must be either an \\
\tinteger, a real number or a probability distribution.\";

\t
EvolveForTimeTau[{time_, 
                  stateBeforeCursorObsvd_, 
                  cursorNotAtKPlus1_, 
                  stateAfterCursorObsvd_
                 }, m_, k_, circuit_, tau_]:=
\tModule[{new},
\t\tnew = SchrodingerEvolution[stateAfterCursorObsvd,m,k,circuit,tau];
\t\tJoin[{time+tau, new}, ReadCursorBits[k+1,new] ]
\t]

(* Explain result creates a narrative that explains the output
   from EvolveQC
*)
ExplainResults[results_]:=
\t(SetOptions[$Output, PageWidth->200];
\t Scan[ExplainResult, results])
\t
ExplainResult[{t_, stateIn_, cursorFoundAt_, stateOut_}]:=
\t(Print[\"Time t=\", t];
\t Print[\"State of QC = \", stateIn];
\t Print[\"Cursor observed at position = \", cursorFoundAt];
\t Print[\"Collapsed state of QC = \", stateOut];
\t Print[\"\\n\"]
\t)
\t\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["EvolveQC[ket[1,0,0,0], sqrtNOTcircuit]", "Input",
  AspectRatioFixed->True]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Plotting the Evolution", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
You can visualize the time evolution of a Feynman quantum computer \
using the function PlotEvolution.  PlotEvolution shows the probabilities of \
obtaining each possible state for the register at the cursor observation \
times, until the computation is done.

To use PlotEvolution you must first generate a particular evolution using \
EvolveQC.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["?PlotEvolution", "Input",
  AspectRatioFixed->True],

Cell[BoxData[
    \("PlotEvolution[evolution] draws a graphic that illustrates the time \
evolution of the memory register of a Feynman quantum computer. It takes a \
single argument, the output from EvolveQC, and plots the probability (i.e. \
|amplitude|^2) of obtaining each eigenstate of the memory register at the \
times when the cursor is observed. By default PlotEvolution only plots the \
probabilities that prevail immediately before the cursor is measured.  You \
can give the option AfterObservingCursor->True to see the effect of measuring \
the cursor on the relative probabilities of finding the memory register in \
each of its possible states."\)], "Print"]
}, Open  ]],

Cell["\<\
PlotEvolution takes a single input, the output of the function \
EvolveQC, and returns a graphic that shows the probability (i.e. | amplitude \
|^2) of each eigenstate of the memory register at the times when the cursor \
is observed. For compactness of notation we label the eigenstates of the (in \
this case 4-bit) memory register, |i>,  in base 10 notation.   For example, \
|5> corresponds to the eigenstate of the memory register that is really \
|0101> and |15> corresponds to the eigenstate of the memory register that is \
really |1111>. The vertical axis shows that probability of obtaining that \
eigenstate if the memory register were to be measured at the given time. \
Notice that there is a zero probability of ever obtaining certain eigenstates \
showing that certain configurations of the memory register are forbidden.
\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
We seed the pseudo-random number generator just to ensure \
reproducible results.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
SeedRandom[1234];
evoln = EvolveQC[ket[1,0,0,0], sqrtNOTcircuit];
PlotEvolution[evoln]\
\>", "Input",
  AspectRatioFixed->True],

Cell["\<\

PlotEvolution can take two options that controls the information that is \
output. By default, PlotEvolution plots the probabilities of finding the \
computer in the various eigenstates at times t1, t2, t3 etc before the cursor \
position is observed. By setting the option AfterObservingCursor->True you \
can plot the probability of finding the computer in the various eigenstates \
both before and after the cursor position is observed. Hence you can \
visualize the effect of the cursor measurement operations by setting the \
option AfterObservingCursor->True.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
SeedRandom[1234];
evoln = EvolveQC[ket[1,0,0,0], sqrtNOTcircuit];
PlotEvolution[evoln, AfterObservingCursor->True]\
\>", "Input",
  AspectRatioFixed->True],

Cell["\<\
The time intervals between measurements need not be regular.  Try \
setting the time between observations, in EvolveQC, to a probability \
distribution using the option TimeBetweenObservations.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
SeedRandom[1234];
evoln = EvolveQC[ket[1,0,0,0], sqrtNOTcircuit, 
                 TimeBetweenObservations->NormalDistribution[1,0.7]];
PlotEvolution[evoln, AfterObservingCursor->True]\
\>", "Input",
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Code", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
Options[PlotEvolution] = {BeforeObservingCursor->True, 
                          AfterObservingCursor->False};

PlotEvolution[evolnData_, opts___]:=
\tModule[{opt1,opt2,times, probsB, probsA, triplesB,triplesA, graphicToPlot},
\t\topt1  = BeforeObservingCursor /. {opts} /. Options[PlotEvolution];
\t\topt2  = AfterObservingCursor  /. {opts} /. Options[PlotEvolution];
\t\ttimes = Map[#[[1]]&, evolnData];
\t\tprobsB = Map[Probabilities[ #[[2]] ]&, evolnData];
\t\tprobsA = Map[Probabilities[ #[[4]] ]&, evolnData];
\t\t
\t\ttriplesB = MapThread[BuildTriple[#1, #2]&, {times, probsB}];
\t\ttriplesA = MapThread[BuildTriple[#1, #2]&, {times, probsA}];
\t\t
\t\tgraphicToPlot =
\t\t\tWhich[opt1===True && opt2===False,
\t\t\t\t\t{Graphics3D[{Thickness[0.01], GrayLevel[0.5], triplesB}]},
\t\t          opt1===True && opt2===True,
\t\t          \t{Graphics3D[{Thickness[0.01], GrayLevel[0.5], triplesB}],
\t\t          \t Graphics3D[{Thickness[0.01], RGBColor[1,0,0], triplesA}]},
\t\t          opt1===False && opt2===True,
\t\t          \t{Graphics3D[{Thickness[0.01], RGBColor[1,0,0], triplesA}]},
\t\t          opt1===False && opt2===False,
\t\t      \t     Message[PlotEvolution::plottingOff]
\t\t         ];
\t\tShow[graphicToPlot,
\t\t\t PlotRange->{0,1},
\t         Axes->True,
\t         BoxRatios->{4,3,2},
\t         Boxed->False,
\t         AxesEdge->{{-1,-1},{1,-1},{1,1}},
\t         AxesLabel->{\"time\", \"|i>\", \"Pr(|i>)\"},
\t         FaceGrids->{{0,0,-1}, {0,1,0}, {-1,0,0}}]
\t]

PlotEvolution::usage =
\"PlotEvolution[evolution] draws a graphic that illustrates the \\
time evolution of the memory register of a Feynman quantum \\
computer. It takes a single argument, the output from EvolveQC, \\
and plots the probability (i.e. |amplitude|^2) of obtaining each \\
eigenstate of the memory register at the times when the cursor \\
is observed. By default PlotEvolution only plots the probabilities \\
that prevail immediately before the cursor is measured.  You can \\
give the option AfterObservingCursor->True to see the effect of \\
measuring the cursor on the relative probabilities of finding the \\
memory register in each of its possible states.\";

PlotEvolution::plottingOff =
\t\"You have turned off both plotting options so no graphics output \\
\twill be produced. You should set BeforeObservingCursor->True or \\
\tAfterObservingCursor->True or both.\";

BuildTriple[time_, prob_]:=
\tLine[MapIndexed[MakeStep[Flatten[{time, #2-{1}, #1}], 0.75]&, prob]]
\t
MakeStep[{time_, center_, height_}, width_]:=
\tSequence[{time, center-width/2., 0},
\t         {time, center-width/2., height},
\t         {time, center+width/2., height},
\t         {time, center+width/2., 0}
\t]
\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Extracting an Answer", "Section",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
The answer is extracted by reading just the program qubits of the \
final state of the memory register.  To figure out which qubits are program \
qubits, we need to know the number of lines in the circuit (i.e. m) and the \
number of gates (i.e. k).  The last m qubits in the register are the program \
qubits and the first k+1 qubits in the register are the cursor qubits. For \
the sqrtNOTcircuit, m=1 and k=2.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
The state ket[1,0,0,0] represents an input of 0 to the square root \
of NOT squared circuit (the 4th bit is a 0, the first 3 bits are cursor bits \
with the cursor initialized at its start position). So we expect to see the \
answer 1 because the SqtNOT(SqrtNOT(0)) = NOT(0) = 1.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
        input = ket[0];
       cursor = ket[1,0,0];
 initialState = Direct[cursor, input];
            m = 1; 
            k = 2;
        evoln = EvolveQC[initialState, sqrtNOTcircuit];
       output = ExtractAnswer[m,k,evoln]\
\>", "Input",
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["Conversely, if we input a 1 we expect to see an output of 0.", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
        input = ket[1];
       cursor = ket[1,0,0];
 initialState = Direct[cursor, input];
            m = 1; 
            k = 2;
        evoln = EvolveQC[initialState, sqrtNOTcircuit];
       output = ExtractAnswer[m,k,evoln]\
\>", "Input",
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
If we input a 0 and a 1 simultaneously the output ought to be \
equally likely to be 0 or 1. Try running this experiment several times.\
\>", 
  "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
        input = 1/Sqrt[2] (ket[0] + ket[1]);
       cursor = ket[1,0,0];
 initialState = Direct[cursor, Expand[input]];
            m = 1; 
            k = 2;
        evoln = EvolveQC[initialState, sqrtNOTcircuit];
       output = ExtractAnswer[m,k,evoln]\
\>", "Input",
  AspectRatioFixed->True],

Cell["\<\
The final state of the register is a superposition that weights the \
probability of obtaining a 0 as high as that of obtaining a 1. For example, \
take a particular final state and imagine measuring 20 identical copies of \
this state (this is just a thought experiment, you cannot copy an arbitrary \
quantum state exactly).\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["Table[ExtractAnswer[m,k,evoln], {20}]", "Input",
  AspectRatioFixed->True],

Cell["\<\
 inputBit = 1/Sqrt[2] (ket[0] + ket[1]);
 cursor = ket[1,0,0];
 initialState = Direct[cursor, inputBit]
     m = 1; 
     k = 2;
 evoln = EvolveQC[1/Sqrt[2] ket[1,0,0,0] + 1/Sqrt[2] ket[1,0,0,1], 
                  sqrtNOTcircuit];
output = ExtractAnswer[m,k,evoln]\
\>", "Input",
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Code", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
ExtractAnswer[m_, k_, evolution_]:=
\tModule[{bitsRead},
\t\tbitsRead = 
\t\t\tReadPartOfMemoryRegister[Last[evolution][[4]], 
\t                                 Table[i, {i, k+2, m+k+1}]][[1]];
\t    Apply[StringJoin, Map[ToString, bitsRead]]
\t]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell["\<\
(*==========================*)
(* ReadPartOfMemoryRegister *)
(*==========================*)
(* Given a superposition representing the state of the memory of a
   quantum computer, return the result of measuring a specific
   subset of the qubits in the memory.
   
   >>> This function is used in error correcting codes <<<
*)
Options[ReadPartOfMemoryRegister] = {TraceProgress->False};

ReadPartOfMemoryRegister[w_. ket[bits__] + kets_., bitsToRead_, opts___]:=
\tModule[{nBits, superposition, resultsPerStep, traceQ},
\t\ttraceQ = TraceProgress /. {opts} /. Options[ReadPartOfMemoryRegister];
\t\tnBits = Length[{bits}]; (* figure out number of bits in memory *)
\t\tsuperposition = {\"BeforeAnyMeasurements\", w ket[bits] + kets};
\t\tresultsPerStep = FoldList[MeasureIthBit[#2,#1,nBits]&,
\t\t                          superposition,
\t\t                          bitsToRead
\t\t                         ];
\t\tWhich[traceQ===False, 
\t\t        {Rest[Map[#[[1]]&, resultsPerStep]],  (* results for bits *)
\t\t         Last[resultsPerStep][[2]]            (* projected state *)
\t\t        },
\t\t      traceQ===True,
\t\t        ColumnForm[resultsPerStep] (*list of {results,projectedStates}*)
\t\t     ]
\t]
\t
ReadPartOfMemoryRegister::usage =
\t\"ReadPartOfMemoryRegister[superposition, bitsToRead] reads the state \\
\tof selected bits in the memory register. As the i-th and j-th bit \\
\tmeasurement operators commute (for any i and j), it does not matter \\
\tin what order you measure the bits.\";\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Closed]],

Cell[CellGroupData[{

Cell["", "Subsubsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["On[General::spell1]", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Bras and Kets defined in here ...", "Section",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["Converting Kets to Column Vectors", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
KetToColumnVector[ket[0]]:={{1},{0}}      (* spin up   = 0 *)
KetToColumnVector[ket[1]]:={{0},{1}}      (* spin down = 1 *)
KetToColumnVector[ket[bits__]]:=
\tApply[Direct, Map[KetToColumnVector[ket[#]]&, {bits}]]

KetToColumnVector[a_ ket_ket]:=
\ta KetToColumnVector[ket]
\t
KetToColumnVector[Plus[ket_, kets___]]:=
\tApply[Plus, Map[KetToColumnVector, {ket, kets}]]
\t
KetToColumnVector[superposition_]:=
\tKetToColumnVector[ Expand[superposition] ] \
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Converting Bras to Row Vectors", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["\<\
BraToRowVector[bra[0]]:={{1,0}}
BraToRowVector[bra[1]]:={{0,1}}
BraToRowVector[w_. bra[bits__]]:=
\tw * Apply[Direct, Map[BraToRowVector[bra[#]]&, {bits}]]
BraToRowVector[w_. bra[bits__] + bras_.]:=
\tBraToRowVector[w * bra[bits]] + BraToRowVector[bras]
BraToRowVector[superposition_]:=
\tBraToRowVector[Expand[superposition]]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell[BoxData[
    \(General::"spell1" \( : \ \) 
      "Possible spelling error: new symbol name \"\!\(bras\)\" is similar to \
existing symbol \"\!\(bra\)\"."\)], "Message"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Converting Column Vectors to Kets", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["\<\
ColumnVectorToKet[amplitudes_]:=
\tApply[Plus,
\t\t  MapThread[(#1[[1]] #2)&,
\t\t            {amplitudes,
\t\t             EigenKets[ Length[amplitudes] ]
\t\t            }
\t\t           ]
\t\t ]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell[BoxData[
    \(General::"spell1" \( : \ \) 
      "Possible spelling error: new symbol name \"\!\(amplitudes\)\" is \
similar to existing symbol \"\!\(Amplitudes\)\"."\)], "Message"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Converting Row Vectors To Bras", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
RowVectorToBra[{{wi__}}]:=
\tModule[{eigenBras},
\t\teigenBras = EigenKets[Length[{wi}]] /. ket->bra;
\t\tApply[Plus, MapThread[(#1 #2)&, {{wi}, eigenBras}]]
\t]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Converting Between Bras and Kets", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
KetToBra[ket_]:=
\tRowVectorToBra[Conjugate[Transpose[KetToColumnVector[ket]]]]

BraToKet[bra_]:=
\tColumnVectorToKet[Conjugate[Transpose[BraToRowVector[bra]]]]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Average Value of an Observable", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
ExpectationValue[w_. ket[bits__] + kets_., observable_]:=
\t(If[!HermitianQ[observable], 
\t\t(Message[ExpectationValue::notHermitian]; Abort[]),
\t\tIf[Length[observable] != 2^Length[{bits}],
\t\t   (Message[ExpectationValue::incompatible]; Abort[])]];
\t\t       
\t (BraToRowVector[KetToBra[w * ket[bits] + kets]] . 
\t  observable . 
\t  KetToColumnVector[w * ket[bits] + kets]
\t )[[1,1]]  (* scalar = a 1 x 1 matrix, [[1,1]] removes the parentheses *)
\t)

ExpectationValue[superposition_, observable_]:=
\tExpectationValue[Expand[superposition], observable]

ExpectationValue::notHermitian =
\t\"Your purported observable is not an Hermitian matrix.\";
ExpectationValue::incompatible =
\t\"The dimensions of the state vector and observable are incompatible.\";
\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Creating Eigenstates that Span a Hilbert Space", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
BasisEigenstates[m_Integer]:= EigenKets[2^m]

BasisEigenstates::usage = 
  \"BasisEigenstates[m] returns the complete set of \\
  eigenstates that span the Hilbert space of an m-bit \\
  quantum memory register.\";\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell["\<\
EigenKets[n_]:=
\tModule[{bits},
\t\tbits = Table[Apply[ket, IntegerDigits[i,2]], 
\t\t             {i, 0, n-1}];
\t\t          (* last eigenket has the most bits *)
\t\tMap[PadTo[Length[Last[bits]], #]&, bits]
\t]

PadTo[nDigits_, digits_]:=
\tJoin[Apply[ket, Table[0,{nDigits - Length[digits]}]], 
\t     digits]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Accessing Amplitudes of Superpositions and Computing \
Probabilities\
\>", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
Options[Amplitudes] = {ShowEigenstates->False};

ShowEigenstates::usage = 
\t\"ShowEigenstates is an option for Amplitudes that \\
\tdetermines whether the 
output should be a list of the \\
\tamplitudes or a list of {eigenstate, 
amplitude} pairs.\";

Amplitudes[w_. ket[bits__] + kets_., opts___]:=
\tModule[{showeigen},
\tshoweigen = ShowEigenstates /. {opts} /. Options[Amplitudes];
\tWhich[showeigen == True, 
\t\t\tMap[{#, Coefficient[w ket[bits] + kets, #]}&,
\t\t        BasisEigenstates[ Length[{bits}] ]
\t\t       ],
\t\t  showeigen == False,
\t\t    Map[Coefficient[w ket[bits] + kets, #]&,
\t\t        BasisEigenstates[ Length[{bits}] ]
\t\t       ]
\t\t  ]
\t]

(* This clause catches cases like 1/Sqrt[2] (ket[0] + ket[1]) etc *)\t
Amplitudes[c_ (w_. ket[bits__] + kets_.)]:=
\tAmplitudes[ Expand[c (w ket[bits] + kets)] ]

Amplitudes::usage = 
  \"Amplitudes[superposition] returns the amplitudes of the \\
  eigenstates in a superposition or ket vectors.\";\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell["\<\
Options[Probabilities] = {ShowEigenstates->False};

Probabilities[w_. ket[bits__] + kets_., opts___]:=
\tModule[{showeigen, amplitudes, symbols, sumOfSquares},
\t\tshoweigen    = ShowEigenstates /. {opts} /. Options[Probabilities];
\t\tamplitudes   = Amplitudes[w ket[bits] + kets];
\t\tsymbols      = SymbolicCoefficients[amplitudes]; (*see below*)
\t\tsumOfSquares = Simplify[
\t\t                Apply[Plus, 
\t\t                      Map[ComplexExpand[Abs[#]^2, symbols]&, 
\t\t                          amplitudes]]];
\t\tamplitudes   = If[sumOfSquares=!=1,  (* renormalize amplitudes
\t\t                                         if necessary *)
\t\t                  amplitudes/Sqrt[sumOfSquares],
\t\t                  amplitudes];
\t\tWhich[showeigen == True,  
\t\t       MapThread[{#1, ComplexExpand[Abs[#2]^2, symbols]}&, 
\t\t                 {BasisEigenstates[Length[{bits}]], amplitudes}
\t\t                ],
\t\t\t  showeigen == False, 
\t\t\t   Map[ComplexExpand[Abs[#]^2, symbols]&, amplitudes]
\t    ]
\t]

Probabilities[c_ (w_. ket[bits__] + kets_.)]:=
\tProbabilities[ Expand[c (w ket[bits] + kets)] ]

Probabilities::usage =
\t\"Probabilities[superposition] returns the probabilities of \\
\t finding a system in a state described by superposition in \\
\t each of its possible eigenstates upon being measured (observed). \\
\t If Probabilities is given the option ShowEigenstates->True \\
\t the function returns a list of {eigenstate, probability} pairs.\";\
\>", 
  "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell["\<\
SymbolicCoefficients[amplitudes_List]:=
\tSelect[Union[Flatten[Map[Variables, amplitudes]]], 
\t\t   Not[MatchQ[#, Abs[_]]]&]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Testing Whether a Ket is Properly Normalized", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
Needs[\"Algebra`ReIm`\"];

NormalizedKetQ[ket_]:=
\tModule[{columnVector},
\t\tcolumnVector = KetToColumnVector[ket];
\t\t(Inner[Times, 
\t\t       Conjugate[Transpose[columnVector]], 
               columnVector,
               Plus
              ] == {{1}} // N ) /. z_ Conjugate[z_] :> Abs[z]^2
    ]
   
NormalizedKetQ::usage =
\t\"NormalizedKetQ[ket] returns True if the square \\
\tmoduli of the amplitudes of the eigenkets in the \\
\tsuperposition \\\"ket\\\" sum to 1. If \\\"ket\\\" has non-numeric \\
\tamplitudes, the normalization cannot always be determined.\";\
\>", "Input",\

  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["NormalizeKet", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
NormalizeKet[superposition_]:=
\tsuperposition /; NormalizedKetQ[superposition]
NormalizeKet[superposition_]:=
\tExpand[superposition / 
\t       Sqrt[Apply[Plus, 
\t                  Map[Abs[#]^2&, 
\t                      Amplitudes[superposition, 
\t                                 ShowEigenstates->False]
\t                     ]
\t                 ]
\t           ]
\t]
\t      
NormalizeKet::usage =
\t\"NormalizeKet[superposition] is used to normalize a given \\
\tsuperposition of
 ket vectors. That is, if the sum of the squares \\
\tof the absolute values of 
the amplitudes of the eigenstates in \\
\tthe superposition do not sum to 1, 
NormalizeKet rescales the \\
\tamplitudes so that they squared moduli will sum 
to 1.\";\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Direct Product", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
(* Last modified 09/07/96 *)
Needs[\"LinearAlgebra`MatrixManipulation`\"];

Direct[op1_, op2_]:=
\tBlockMatrix[Outer[Times, op1, op2]] /; MatrixQ[op1] && MatrixQ[op2]
\t
Direct[ket_, bra_]:=
\tDirect[KetToColumnVector[ket], BraToRowVector[bra]] /; IsKetQ[ket] && \
IsBraQ[
bra]
\t
Direct[ket1_, ket2_]:=
\tColumnVectorToKet[
\t\tDirect[KetToColumnVector[ket1],
\t           KetToColumnVector[ket2]]
\t]/; IsKetQ[ket1] && IsKetQ[ket2]

Direct[bra1_, bra2_]:=
\tRowVectorToBra[
\t\tDirect[BraToRowVector[bra1],
\t\t\t   BraToRowVector[bra2]]
\t] /; IsBraQ[bra1] && IsBraQ[bra2]
\t
Direct[bra_, ket_]:=
\t(Message[Direct::braket];
\t Direct[BraToRowVector[bra], KetToColumnVector[ket]]) /; IsBraQ[bra] && 
IsKetQ[ket]

Direct[bra_, op_]:=
\t(Message[Direct::braop];
\t Direct[BraToRowVector[bra], op]) /; IsBraQ[bra] && MatrixQ[op]
\t
Direct[op_, bra_]:=
\t(Message[Direct::opbra];
\t Direct[op, BraToRowVector[bra]]) /; MatrixQ[op] && IsBraQ[bra]
\t
Direct[ket_, op_]:=
\t(Message[Direct::ketop];
\t Direct[KetToColumnVector[ket], op]) /; IsKetQ[ket] && MatrixQ[op]
\t
Direct[op_, ket_]:=
\t(Message[Direct::opket];
\t Direct[op, KetToColumnVector[ket]]) /; MatrixQ[op] && IsKetQ[ket]

Direct[matrices__]:=
\tFold[Direct, First[{matrices}], Rest[{matrices}]]

Direct::braket =
\t\"Warning - You are taking the DIRECT product of a bra \\
\tand a ket. This is 
unusual. Perhaps you meant to use \\
\tthe DOT product?\";
\t
Direct::braop =
\t\"Warning - You are taking the DIRECT product of a bra \\
\twith an operator. 
This is unusual. Perhaps you meant to use \\
\tthe DOT product?\";
\t
Direct::opbra =
\t\"Warning - You are taking the DIRECT product of an operator \\
\twith a bra. 
This is unusual. Perhaps you meant to use \\
\tthe DOT product?\";

Direct::ketop =
\t\"Warning - You are taking the DIRECT product of a ket \\
\twith an operator. 
This is unusual. Perhaps you meant to use \\
\tthe DOT product?\";

Direct::opket =
\t\"Warning - You are taking the DIRECT product of an operator \\
\twith a ket. 
This is unusual. Perhaps you meant to use \\
\tthe DOT product?\";


IsKetQ[w_. ket[__] + kets_.]:= True
IsKetQ[_]:=False
\t
IsBraQ[w_. bra[__] + bras_.]:= True
IsBraQ[_]:=False\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Truth Table of a Logic Gate", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
TruthTable[gate_]:=
\tModule[{n,m},
\t\t{n,m} = Dimensions[gate];
\t\tWhich[Not[n==m && IntegerQ[n] && IntegerQ[m]],
\t\t      Message[TruthTable::notsquare]; Abort[],
\t\t      Not[IntegerQ[Log[2, n]]],
\t\t      Message[TruthTable::powerof2]; Abort[]
\t\t     ];
\t\tMap[(# -> ColumnVectorToKet[gate . KetToColumnVector[#]])&, 
\t\t    EigenKets[n]
\t\t   ]  // ColumnForm
\t]
\t
TruthTable::notsquare = 
  \"Your input is not a square matrix and cannot, therefore, represent a \\
  
reversible logic gate.\";

TruthTable::powerof2 = 
  \"Your input is not a matrix of dimensions (2^m) x (2^m) for integer m \\
  
and cannot, therefore, represent a reversible logic gate that operates \\
  on 
m bits.\";\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Types of Operators (Matrices)", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
HermitianQ[matrix_]:=
\tmatrix == Conjugate[Transpose[matrix]]\
\>", "Input",
  AspectRatioFixed->True],

Cell["\<\
UnitaryQ[matrix_]:=
  Module[{rows, cols},
\t{rows, cols} = Dimensions[matrix];
\tIf[Not[IntegerQ[rows]] || 
\t   Not[IntegerQ[cols]] || 
\t   rows != cols, Message[UnitaryQ::notsquarematrix]];
\t
\t   Chop[Simplify[ComplexExpand[Conjugate[Transpose[matrix]]] - 
\t                 ComplexExpand[Inverse[matrix]]
\t                ]
\t       ] == ZeroMatrix[rows, cols]
  ]

UnitaryQ::notsquarematrix =
  \"Your input is not a square matrix.\";
  
ZeroMatrix[rows_, cols_]:=
\tTable[0, {rows}, {cols}]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Tools for Making Test Superpositions", "Subsection",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
SymbolicSuperposition[m_]:=
\tApply[Plus,
\t      MapThread[(#1 #2)&, 
\t                {SymbolicAmplitudes[m], BasisEigenstates[m]}]
\t]
\t
SymbolicSuperposition::usage =
\t\"SymbolicSuperposition[m] creates a superposition of 2^m \\
\teigenstates whose
 amplitudes are uninstantiated symbols. These \\
\teigenstates represent the 
possible states of an m-bit memory \\
\tregister of a quantum computer. This 
function is useful for \\
\texploring the effects of quantum mechanical 
operations on \\
\tarbitrary superpositions. Note that the general form does not

\tguarentee that the superposition is normalized.\";
\t
SymbolicAmplitudes[m_]:=
\t(Clear[w];
\t Map[ToExpression[\"w\"<>ToString[#]]&, Table[i,{i,0,2^m - 1}]]
\t)\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell["\<\
Options[RandomSuperposition] = {Normalized->True};

RandomSuperposition[m_, opts___]:=
\tModule[{normalized},
\t\tnormalized = Normalized /. {opts} /. Options[RandomSuperposition];
\t\tsuperposition = Apply[Plus,
\t                          MapThread[(#1 #2)&, 
\t                                    {RandomAmplitudes[m],
\t                                     BasisEigenstates[m]}
\t                                   ]
\t                         ];
\t\tWhich[normalized==True, NormalizeKet[superposition],
\t          normalized==False, superposition
\t         ]
\t]

RandomSuperposition::usage =
\t\"RandomSuperposition[m] creates a normalized superposition \\
\tof 2^m eigenstates whose amplitudes are random complex numbers. \\
\tThese eigenstates represent the possible states of an m-bit \\
\tmemory register of a quantum computer. You can generate an \\
\tun-normalized superposition by setting the option Normalized->False.\";
\t
(* You can pick the amplitudes according to whatever distribution
   you like. In the current case we pick random complex numbers
   uniformly from the square in the complex plane bounded by a lower
   left corner at (-1,-I) and an upper right corner at (1,I).
*)
RandomAmplitudes[m_]:=
\tTable[Random[Complex, {-1-I, 1+I}], {2^m}]\
\>", "Input",
  InitializationCell->True,
  AspectRatioFixed->True],

Cell[BoxData[
    \(General::"spell1" \( : \ \) 
      "Possible spelling error: new symbol name \"\!\(normalized\)\" is \
similar to existing symbol \"\!\(Normalized\)\"."\)], "Message"]
}, Open  ]]
}, Open  ]]
}, Closed]]
}, Open  ]]
},
FrontEndVersion->"Macintosh 3.0",
ScreenRectangle->{{0, 640}, {0, 460}},
AutoGeneratedPackage->Automatic,
WindowToolbars->{},
CellGrouping->Manual,
WindowSize->{614, 413},
WindowMargins->{{10, Automatic}, {7, Automatic}},
PrintingCopies->1,
PrintingPageRange->{1, Automatic},
PrivateNotebookOptions->{"ColorPalette"->{RGBColor, -1}},
ShowCellLabel->True,
ShowCellTags->False,
RenderingOptions->{"ObjectDithering"->True,
"RasterDithering"->False},
MacintoshSystemPageSetup->"\<\
00<0004/0B`000002n88o?mooh<<H`Ui0fL5:0?l0080004/0B`000000]P2:001
0000I00000400`<300000BL?00400@BU_l0000000000060801T1T000000@0000
00000000004000000000000000000000\>"
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1731, 51, 126, 3, 136, "Title",
  Evaluatable->False],
Cell[1860, 56, 107, 3, 53, "Subsubtitle",
  Evaluatable->False],
Cell[1970, 61, 110, 3, 25, "Input",
  InitializationCell->True],

Cell[CellGroupData[{
Cell[2105, 68, 102, 3, 50, "Section",
  Evaluatable->False],
Cell[2210, 73, 905, 19, 222, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[3152, 97, 94, 3, 50, "Section",
  Evaluatable->False],
Cell[3249, 102, 723, 13, 126, "Text",
  Evaluatable->False],
Cell[3975, 117, 188, 6, 30, "Text",
  Evaluatable->False],
Cell[4166, 125, 92, 6, 51, "Input"],
Cell[4261, 133, 281, 7, 62, "Text",
  Evaluatable->False],
Cell[4545, 142, 82, 5, 38, "Input"],
Cell[4630, 149, 436, 11, 110, "Text",
  Evaluatable->False],
Cell[5069, 162, 99, 5, 38, "Input"],
Cell[5171, 169, 314, 8, 62, "Text",
  Evaluatable->False],
Cell[5488, 179, 75, 2, 25, "Input"],
Cell[5566, 183, 298, 8, 62, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[5901, 196, 109, 2, 50, "Section",
  Evaluatable->False],
Cell[6013, 200, 437, 9, 78, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[6475, 213, 96, 2, 46, "Subsection",
  Evaluatable->False],
Cell[6574, 217, 109, 5, 38, "Input",
  InitializationCell->True],
Cell[6686, 224, 147, 7, 64, "Input",
  InitializationCell->True],
Cell[6836, 233, 287, 11, 116, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[7160, 249, 105, 2, 46, "Subsection",
  Evaluatable->False],
Cell[7268, 253, 270, 10, 103, "Input",
  InitializationCell->True],
Cell[7541, 265, 270, 11, 116, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[7848, 281, 125, 2, 46, "Subsection",
  Evaluatable->False],
Cell[7976, 285, 464, 9, 78, "Text",
  Evaluatable->False],
Cell[8443, 296, 523, 22, 259, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[9003, 323, 103, 2, 46, "Subsection",
  Evaluatable->False],
Cell[9109, 327, 116, 2, 30, "Text",
  Evaluatable->False],
Cell[9228, 331, 3900, 90, 90, 625, 44, "GraphicsData", 
"PostScript", "Graphics",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[13153, 425, 103, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[13259, 429, 310, 9, 70, "Input"],

Cell[CellGroupData[{
Cell[13594, 442, 67, 1, 70, "Input"],
Cell[13664, 445, 782, 49, 70, 625, 44, "GraphicsData", 
"PostScript", "Graphics",
  Evaluatable->False,
  ImageCacheValid->False],
Cell[14449, 496, 131, 7, 70, "Output",
  Evaluatable->False]
}, Open  ]],
Cell[14595, 506, 130, 5, 70, "Input"],

Cell[CellGroupData[{
Cell[14750, 515, 62, 1, 70, "Input"],
Cell[14815, 518, 912, 65, 70, 755, 60, "GraphicsData", 
"PostScript", "Graphics",
  Evaluatable->False,
  ImageCacheValid->False],
Cell[15730, 585, 131, 7, 70, "Output",
  Evaluatable->False]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{
Cell[15910, 598, 73, 2, 28, "Subsubsection",
  Evaluatable->False],
Cell[15986, 602, 538, 18, 48, "Text",
  Evaluatable->False],
Cell[16527, 622, 284, 11, 116, "Input",
  InitializationCell->True],
Cell[16814, 635, 254, 6, 46, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[17105, 646, 80, 2, 42, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[17210, 652, 71, 1, 25, "Input"],
Cell[17284, 655, 968, 32, 57, "Output"]
}, Open  ]],
Cell[18267, 690, 198, 5, 46, "Text",
  Evaluatable->False],
Cell[18468, 697, 5717, 126, 78, 755, 60, "GraphicsData", 
"PostScript", "Graphics",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[24222, 828, 80, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[24305, 832, 90, 1, 25, "Input"],
Cell[24398, 835, 185, 5, 30, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[24620, 845, 80, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[24703, 849, 88, 1, 25, "Input"],
Cell[24794, 852, 56, 1, 25, "Input"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[24911, 860, 108, 2, 50, "Section",
  Evaluatable->False],
Cell[25022, 864, 812, 13, 126, "Text",
  Evaluatable->False],
Cell[25837, 879, 125, 2, 25, "Input",
  InitializationCell->True],
Cell[25965, 883, 775, 16, 158, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[26765, 903, 80, 2, 42, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[26870, 909, 240, 9, 90, "Input"],
Cell[27113, 920, 279, 7, 69, "Output"]
}, Open  ]],
Cell[27407, 930, 177, 5, 30, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[27621, 940, 80, 2, 42, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[27726, 946, 217, 9, 90, "Input"],
Cell[27946, 957, 807, 13, 213, "Output"]
}, Open  ]],
Cell[28768, 973, 180, 6, 62, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[28985, 984, 80, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[29068, 988, 62, 1, 25, "Input"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[29179, 995, 110, 2, 50, "Section",
  Evaluatable->False],
Cell[29292, 999, 528, 10, 78, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[29857, 1014, 101, 2, 50, "Section",
  Evaluatable->False],
Cell[29961, 1018, 524, 10, 78, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[30510, 1032, 55, 1, 25, "Input"],
Cell[30568, 1035, 246, 3, 52, "Print"]
}, Open  ]],

Cell[CellGroupData[{
Cell[30851, 1043, 80, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[30934, 1047, 203, 6, 51, "Input"]
}, Open  ]],

Cell[CellGroupData[{
Cell[31174, 1058, 77, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[31254, 1062, 579, 17, 70, "Input",
  InitializationCell->True]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{
Cell[31882, 1085, 107, 2, 30, "Section",
  Evaluatable->False],
Cell[31992, 1089, 649, 11, 94, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[32666, 1104, 55, 1, 25, "Input"],
Cell[32724, 1107, 229, 3, 52, "Print"]
}, Open  ]],
Cell[32968, 1113, 255, 6, 46, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[33248, 1123, 77, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[33328, 1127, 412, 11, 70, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[33777, 1143, 80, 2, 28, "Subsubsection",
  Evaluatable->False],
Cell[33860, 1147, 187, 5, 51, "Input"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[34096, 1158, 122, 2, 50, "Section",
  Evaluatable->False],
Cell[34221, 1162, 566, 10, 94, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[34812, 1176, 64, 1, 25, "Input"],
Cell[34879, 1179, 403, 5, 97, "Print"]
}, Open  ]],

Cell[CellGroupData[{
Cell[35319, 1189, 77, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[35399, 1193, 699, 15, 70, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[36135, 1213, 80, 2, 28, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[36240, 1219, 167, 4, 38, "Input"],
Cell[36410, 1225, 458, 16, 39, "Output",
  Evaluatable->False]
}, Open  ]],
Cell[36883, 1244, 98, 2, 30, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[37018, 1251, 80, 2, 42, "Subsubsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[37123, 1257, 60, 1, 25, "Input"],
Cell[37186, 1260, 110, 6, 24, "Output",
  Evaluatable->False]
}, Open  ]],
Cell[37311, 1269, 180, 5, 30, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[37528, 1279, 80, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[37611, 1283, 167, 4, 38, "Input"],
Cell[37781, 1289, 98, 2, 30, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[37916, 1296, 96, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[38015, 1300, 793, 14, 70, "Text",
  Evaluatable->False]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{
Cell[38857, 1320, 125, 2, 30, "Section",
  Evaluatable->False],
Cell[38985, 1324, 2035, 30, 318, "Text",
  Evaluatable->False],
Cell[41023, 1356, 1879, 29, 286, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[42927, 1389, 58, 1, 25, "Input"],
Cell[42988, 1392, 757, 10, 172, "Print"]
}, Open  ]],
Cell[43760, 1405, 556, 10, 94, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[44341, 1419, 77, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[44421, 1423, 1722, 35, 70, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[46180, 1463, 80, 2, 28, "Subsubsection",
  Evaluatable->False],
Cell[46263, 1467, 225, 6, 64, "Input"],
Cell[46491, 1475, 287, 7, 46, "Text",
  Evaluatable->False],
Cell[46781, 1484, 292, 11, 24, "Output",
  Evaluatable->False],
Cell[47076, 1497, 486, 9, 78, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[47599, 1511, 80, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[47682, 1515, 97, 2, 30, "Text",
  Evaluatable->False],
Cell[47782, 1519, 54, 1, 25, "Input"],
Cell[47839, 1522, 49, 1, 25, "Input"],
Cell[47891, 1525, 222, 6, 46, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[48150, 1536, 80, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[48233, 1540, 143, 5, 30, "Text",
  Evaluatable->False],
Cell[48379, 1547, 224, 6, 64, "Input"],
Cell[48606, 1555, 107, 2, 30, "Text",
  Evaluatable->False],
Cell[48716, 1559, 62, 1, 25, "Input"],
Cell[48781, 1562, 150, 5, 30, "Text",
  Evaluatable->False],
Cell[48934, 1569, 68, 1, 25, "Input"],
Cell[49005, 1572, 165, 5, 30, "Text",
  Evaluatable->False],
Cell[49173, 1579, 60, 1, 25, "Input"]
}, Open  ]],

Cell[CellGroupData[{
Cell[49270, 1585, 96, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[49369, 1589, 1148, 20, 190, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[50554, 1614, 133, 3, 46, "Subsection",
  Evaluatable->False],
Cell[50690, 1619, 302, 7, 46, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[51017, 1630, 62, 1, 25, "Input"],
Cell[51082, 1633, 259, 4, 67, "Print"]
}, Open  ]],
Cell[51356, 1640, 310, 7, 62, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[51691, 1651, 77, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[51771, 1655, 1417, 35, 70, "Input",
  InitializationCell->True],
Cell[53191, 1692, 841, 23, 70, "Input",
  InitializationCell->True],
Cell[54035, 1717, 1094, 29, 70, "Input",
  InitializationCell->True],
Cell[55132, 1748, 1377, 44, 70, "Input",
  InitializationCell->True]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{
Cell[56558, 1798, 114, 2, 30, "Subsection",
  Evaluatable->False],
Cell[56675, 1802, 880, 14, 142, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[57580, 1820, 52, 1, 25, "Input"],
Cell[57635, 1823, 883, 12, 202, "Print"]
}, Open  ]],

Cell[CellGroupData[{
Cell[58555, 1840, 77, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[58635, 1844, 4294, 110, 70, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[62966, 1959, 80, 2, 28, "Subsubsection",
  Evaluatable->False],
Cell[63049, 1963, 81, 1, 25, "Input"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[63179, 1970, 92, 2, 46, "Subsection",
  Evaluatable->False],
Cell[63274, 1974, 418, 10, 94, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[63717, 1988, 57, 1, 25, "Input"],
Cell[63777, 1991, 673, 9, 157, "Print"]
}, Open  ]],
Cell[64465, 2003, 909, 15, 158, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[65399, 2022, 80, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[65482, 2026, 153, 5, 30, "Text",
  Evaluatable->False],
Cell[65638, 2033, 137, 5, 51, "Input"],
Cell[65778, 2040, 638, 12, 110, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[66453, 2057, 80, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[66536, 2061, 165, 5, 51, "Input"],
Cell[66704, 2068, 265, 6, 46, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[67006, 2079, 80, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[67089, 2083, 235, 6, 64, "Input"]
}, Open  ]],

Cell[CellGroupData[{
Cell[67361, 2094, 77, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[67441, 2098, 2784, 65, 70, "Input",
  InitializationCell->True]
}, Closed]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[70286, 2170, 87, 2, 30, "Section",
  Evaluatable->False],
Cell[70376, 2174, 484, 9, 78, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[70885, 2187, 80, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[70968, 2191, 351, 7, 62, "Text",
  Evaluatable->False],
Cell[71322, 2200, 277, 9, 103, "Input"]
}, Open  ]],

Cell[CellGroupData[{
Cell[71636, 2214, 80, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[71719, 2218, 124, 2, 30, "Text",
  Evaluatable->False],
Cell[71846, 2222, 277, 9, 103, "Input"]
}, Open  ]],

Cell[CellGroupData[{
Cell[72160, 2236, 80, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[72243, 2240, 210, 6, 46, "Text",
  Evaluatable->False],
Cell[72456, 2248, 306, 9, 103, "Input"],
Cell[72765, 2259, 398, 8, 62, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[73200, 2272, 80, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[73283, 2276, 80, 1, 25, "Input"],
Cell[73366, 2279, 316, 10, 116, "Input"]
}, Open  ]],

Cell[CellGroupData[{
Cell[73719, 2294, 77, 2, 42, "Subsubsection",
  Evaluatable->False],
Cell[73799, 2298, 326, 10, 70, "Input",
  InitializationCell->True],
Cell[74128, 2310, 1591, 37, 70, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[75756, 2352, 73, 2, 28, "Subsubsection",
  Evaluatable->False],
Cell[75832, 2356, 90, 2, 25, "Input",
  InitializationCell->True]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[75971, 2364, 100, 2, 50, "Section",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[76096, 2370, 103, 2, 70, "Subsection",
  Evaluatable->False],
Cell[76202, 2374, 532, 16, 70, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[76771, 2395, 100, 2, 70, "Subsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[76896, 2401, 405, 11, 70, "Input",
  InitializationCell->True],
Cell[77304, 2414, 174, 3, 70, "Message"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[77527, 2423, 103, 2, 70, "Subsection",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[77655, 2429, 276, 11, 70, "Input",
  InitializationCell->True],
Cell[77934, 2442, 187, 3, 70, "Message"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[78170, 2451, 100, 2, 70, "Subsection",
  Evaluatable->False],
Cell[78273, 2455, 240, 8, 70, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[78550, 2468, 102, 2, 70, "Subsection",
  Evaluatable->False],
Cell[78655, 2472, 239, 8, 70, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[78931, 2485, 100, 2, 70, "Subsection",
  Evaluatable->False],
Cell[79034, 2489, 847, 23, 70, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[79918, 2517, 116, 2, 70, "Subsection",
  Evaluatable->False],
Cell[80037, 2521, 293, 9, 70, "Input",
  InitializationCell->True],
Cell[80333, 2532, 393, 14, 70, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[80763, 2551, 146, 5, 70, "Subsection",
  Evaluatable->False],
Cell[80912, 2558, 1053, 33, 70, "Input",
  InitializationCell->True],
Cell[81968, 2593, 1564, 37, 70, "Input",
  InitializationCell->True],
Cell[83535, 2632, 204, 6, 70, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[83776, 2643, 114, 2, 70, "Subsection",
  Evaluatable->False],
Cell[83893, 2647, 656, 21, 70, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[84586, 2673, 82, 2, 70, "Subsection",
  Evaluatable->False],
Cell[84671, 2677, 815, 26, 70, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[85523, 2708, 84, 2, 70, "Subsection",
  Evaluatable->False],
Cell[85610, 2712, 2269, 86, 70, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[87916, 2803, 97, 2, 70, "Subsection",
  Evaluatable->False],
Cell[88016, 2807, 785, 27, 70, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[88838, 2839, 99, 2, 70, "Subsection",
  Evaluatable->False],
Cell[88940, 2843, 113, 4, 70, "Input"],
Cell[89056, 2849, 580, 21, 70, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[89673, 2875, 106, 2, 70, "Subsection",
  Evaluatable->False],
Cell[89782, 2879, 809, 27, 70, "Input",
  InitializationCell->True],

Cell[CellGroupData[{
Cell[90616, 2910, 1350, 33, 70, "Input",
  InitializationCell->True],
Cell[91969, 2945, 187, 3, 70, "Message"]
}, Open  ]]
}, Open  ]]
}, Closed]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

